/* porttree.c -- Port tree management
 *
 * This file is part of TUA.
 * 
 *   Copyright (C) 1991,96  Lele Gaifax
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the license, or (at
 *   your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "tua.h"

static BTREE Root = NULL;
static size_t ports_count = 0;

static int
DEFUN (port_compare, (port1, port2),
       CONST port_rec_t * port1 AND CONST port_rec_t * port2)
{
  int ret = strcmp (port1->Port, port2->Port);

  return ret;
}

static PTR
DEFUN (port_new, (data),
       CONST PTR data)
{
  port_rec_t *pa;
  int idx;

  pa = (port_rec_t *) xmalloc (sizeof (port_rec_t));
  pa->Port = savestring (((CONST port_rec_t *) data)->Port);
  pa->Killed = FALSE;
  for (idx = 0; idx < TIMESLICES; idx++)
    pa->Activity[idx] = 0.0;

  return (PTR) pa;
}

port_rec_t *
DEFUN (insert_port, (port),
       CONST char *port)
{
  port_rec_t pa;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) port_compare, (makenew_func_t) port_new);

  /*
   * Reset the number of ports, so the next call to number_of_ports() will
   * recalculate it.
   *
   * Since number_of_ports() is called at report time, it is very probable
   * that no more ports will be added to the tree... but just to be sure ;-)
   */
  ports_count = 0;
  pa.Port = (char *) get_possible_port_alias (port);
  return (port_rec_t *) btree_insert (Root, &pa);
}

void
DEFUN (kill_port, (port),
       CONST char *port)
{
  port_rec_t pa;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) port_compare, (makenew_func_t) port_new);

  pa.Port = (char *) get_possible_port_alias (port);
  
  ((port_rec_t *) (btree_insert (Root, &pa)))->Killed = TRUE;
}

static traverse_func_t originalTraverseFunction;

static void
DEFUN (FilteredEnquiry, (port),
       CONST port_rec_t * port)
{
  if (! port->Killed)
    (*originalTraverseFunction) (port);
}

void
DEFUN (enquire_ports, (funct),
       traverse_func_t funct)
{
  originalTraverseFunction = funct;
  btree_traverse (Root, (traverse_func_t) FilteredEnquiry);
}

port_rec_t *
DEFUN (search_port, (port),
       CONST char *port)
{
  port_rec_t pa;

  pa.Port = (char *) get_possible_port_alias (port);
  return (port_rec_t *) btree_search (Root, &pa);
}

size_t
DEFUN_VOID (number_of_ports)
{
  /* 0 means some ports has been added to the tree since last call */
  if (! ports_count)
    ports_count = btree_count (Root);
  
  return ports_count;
}

#if USE_TCL

int
DEFUN (tcl_kill_port, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 2)
    {
      sprintf (interp->result, _("Wrong number of Parameters: %s needs just one PortName"), argv[0]);
      return TCL_ERROR;
    }
  kill_port (argv[1]);
  return TCL_OK;
}

#endif /* if USE_TCL */
