/* Copyright (c) 2017-2021, The Tor Project, Inc. */
/* See LICENSE for licensing information */

/**
 * \file events.h
 * \brief Header file for Tor tracing instrumentation definition.
 **/

#ifndef TOR_LIB_TRACE_EVENTS_H
#define TOR_LIB_TRACE_EVENTS_H

#include "orconfig.h"

/*
 * A tracepoint signature is defined as follow:
 *
 *    tor_trace(<subsystem>, <event_name>, <args>...)
 *
 * If tracing is enabled, the tor_trace() macro is mapped to all possible
 * instrumentations (defined below). Each instrumentation type MUST define a
 * top level macro (TOR_TRACE_<type>) so it can be inserted into each
 * tracepoint.
 *
 * In case no tracing is enabled (HAVE_TRACING), tracepoints are NOP and thus
 * have no execution cost.
 *
 * Currently, three types of instrumentation are supported:
 *
 *  log-debug: Every tracepoints is mapped to a log_debug() statement.
 *
 *  User Statically-Defined Tracing (USDT): Probes that can be used with perf,
 *  dtrace, SystemTap, DTrace and BPF Compiler Collection (BCC).
 *
 *  LTTng-UST: Probes for the LTTng Userspace Tracer. If USDT interface
 *  (sdt.h) is available, the USDT probes are also generated by LTTng thus
 *  enabling this instrumentation provides both probes.
 */

/** Helper to disambiguate these identifiers in the code base. They should
 * only be used with tor_trace() like so:
 *
 *    tor_trace(TR_SUBSYS(circuit), TR_EV(opened), ...);
 */

#define TR_SUBSYS(name) tor_ ## name
#define TR_EV(name) name

#ifdef HAVE_TRACING

#define tor_trace(subsystem, event_name, ...)                       \
  do {                                                              \
    TOR_TRACE_LOG_DEBUG(subsystem, event_name);             \
    TOR_TRACE_USDT(subsystem, event_name, ## __VA_ARGS__);  \
    TOR_TRACE_LTTNG(subsystem, event_name, ## __VA_ARGS__); \
  } while (0)

/* This corresponds to the --enable-tracing-instrumentation-log-debug
 * configure option which maps all tracepoints to a log_debug() statement. */
#include "lib/trace/debug.h"

/* This corresponds to the --enable-tracing-instrumentation-usdt configure
 * option which will generate USDT probes for each tracepoints. */
#include "lib/trace/usdt/usdt.h"

/* This corresponds to the --enable-tracing-instrumentation-lttng configure
 * option which will generate LTTng probes for each tracepoints. */
#include "lib/trace/lttng/lttng.h"

#else /* !defined(HAVE_TRACING) */

/* Reaching this point, tracing is disabled thus we NOP every tracepoints
 * declaration so we have no execution cost at runtime. */
#define tor_trace(subsystem, name, ...)

#endif /* defined(HAVE_TRACING) */

#endif /* !defined(TOR_LIB_TRACE_EVENTS_H) */
