/*
  Copyright (c) 2006 Paolo Capriotti <p.capriotti@gmail.com>
            (c) 2006 Maurizio Monge <maurizio.monge@kdemail.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

#ifndef PIXMAPLOADER_H
#define PIXMAPLOADER_H

#include <QPixmap>
#include <QString>
#include <QHash>
#include "loader/theme.h"
#include "themeinfo.h"

/**
  * @class PixmapLoader <pixmaploader.h>
  * @brief The simple pixmap loading facility.
  *
  * This is the utility class used by the interface to load pixmap from resource ids.
  * It acts simply as a wrapper around PixmapLoader::ThemeLoader, making it possible to
  * use it in a simple and afficient way.
  *
  * Note that pixmaps will not be all of the specified size, the size is only the base size.
  */
class PixmapLoader {
private:
  class ThemeLoader;
  typedef QHash<ThemeInfo, ThemeLoader*> ThemeLoadersCache;

  /** static cache of the loaders, there should be only one for each (theme,variant) pair  */
  static ThemeLoadersCache& loaders();

  /** the current loader, or NULL */
  ThemeLoader *m_loader;

  /** the current size */
  int m_size;

  /** the current theme */
  ThemeInfo m_theme;

  /** internal, clears references to the currently used loader, if any. */
  void flush();

  /** internal, gets or creates a loader good for the current
      (theme,variant) pair and refs the size */
  void initialize();

public:
  /** constructor */
  PixmapLoader();
  ~PixmapLoader();

  /** set the theme */
  void setTheme(const ThemeInfo& theme);

  /** set the base size of the pixmaps. Note that returned pixmaps's size can be different.
    * For instance, if the size is 100 the "background" generated by the
    * chess variant will be 200x200 (and it will be tiled on the Board)
    */
  void setSize(int s);

  /** looks up a string id (for instance a predefined id, like "background" or "highlighting") */
//   QPixmap operator()(const QString& id);
  QPixmap getPixmap(const QString& id);
  
  QPixmap piecePixmap(const QString& id, bool flipped = false);

  /** returns a value */
  template<typename T>
  T getValue(const QString& id, const ::LuaApi::LuaValueMap* args = NULL, bool allow_nil = false);

  /** returns a static value (not depending on the size) */
  template<typename T>
  T getStaticValue(const QString& id, const ::LuaApi::LuaValueMap* args = NULL, bool allow_nil = false);
};

#endif // PIXMAPLOADER_H
