/* Library libcerf:
 *   Compute complex error functions, based on a new implementation of
 *   Faddeeva's w_of_z. Also provide Dawson and Voigt functions.
 *
 * File statusquotest.c
 *   Test against values generated by libcerf itself.
 *   These tests contribute nothing to guarantee correctness of our computations.
 *   They just protect against future corruption, compiler bugs etc.
 *
 * Copyright:
 *   (C) 2012 Massachusetts Institute of Technology
 *   (C) 2013 Forschungszentrum Jülich GmbH
 *
 * Licence:
 *   ../LICENSE
 *
 * Authors:
 *   Steven G. Johnson, Massachusetts Institute of Technology, 2012, core author
 *   Joachim Wuttke, Forschungszentrum Jülich, 2013, package maintainer
 *
 * Website:
 *   http://apps.jcns.fz-juelich.de/libcerf
 *
 * Revision history:
 *   ../CHANGELOG
 */

#include "cerf.h"
#include "testtool.h"

int main()
{
    result_t result = {0, 0};

    /***************   erfcx   *****************/

    // Further tests, namely erfcx vs Re(cerfcx), are in realtest.c.

    // change of algorithm at -26.7
    RTEST(result, 1e-13, erfcx(-26.7000000001), Inf);
    RTEST(result, 1e-13, erfcx(-26.6999999999), Inf);

    RTEST(result, 1e-13, erfcx(-19.5), 2.76380080745855e+165);
    RTEST(result, 1e-13, erfcx(-12.3), 1.01260988545376e+66);

    // change of algorithm at -6.1
    RTEST(result, 1e-13, erfcx(-6.10000000003), 2.89152976642548e+16);
    RTEST(result, 1e-13, erfcx(-6.10000000001), 2.89152976571995e+16);
    RTEST(result, 1e-13, erfcx(-6.09999999999), 2.89152976501441e+16);
    RTEST(result, 1e-13, erfcx(-6.09999999997), 2.89152976430888e+16);

    RTEST(result, 1e-13, erfcx(-3), 16205.9888539996);
    RTEST(result, 1e-13, erfcx(-1), 5.00898008076228);

    // change of algorithm at 0
    RTEST(result, 1e-13, erfcx(-1e-1), 1.12364335419921);
    RTEST(result, 1e-13, erfcx(-1e-4), 1.00011284791746);
    RTEST(result, 1e-13, erfcx(-1e-8), 1.00000001128379);
    RTEST(result, 1e-13, erfcx(-1e-12), 1.00000000000113);
    RTEST(result, 1e-13, erfcx(-1e-20), 1);
    RTEST(result, 1e-13, erfcx(0), 1);
    RTEST(result, 1e-13, erfcx(1e-20), 1);
    RTEST(result, 1e-13, erfcx(1e-12), 0.999999999998872);
    RTEST(result, 1e-13, erfcx(1e-8), 0.999999988716208);
    RTEST(result, 1e-13, erfcx(1e-4), 0.999887172082539);
    RTEST(result, 1e-13, erfcx(1e-1), 0.896456979969127);

    RTEST(result, 1e-13, erfcx(1), 0.427583576155807);
    RTEST(result, 1e-13, erfcx(3), 0.17900115118139);
    RTEST(result, 1e-13, erfcx(6), 0.0927765678005384);
    RTEST(result, 1e-13, erfcx(10), 0.0561409927438226);
    RTEST(result, 1e-13, erfcx(20), 0.0281743487410513);
    RTEST(result, 1e-13, erfcx(30), 0.0187958888614168);

    // change of algorithm at 50
    RTEST(result, 1e-13, erfcx(49.99999999999), 0.011281536265326);
    RTEST(result, 1e-13, erfcx(50.00000000001), 0.0112815362653215);

    RTEST(result, 1e-13, erfcx(1e2), 0.00564161378298943);
    RTEST(result, 1e-13, erfcx(1e3), 0.000564189301453388);
    RTEST(result, 1e-13, erfcx(1e4), 5.64189580726808e-05);
    RTEST(result, 1e-13, erfcx(1e5), 5.64189583519547e-06);
    RTEST(result, 1e-13, erfcx(1e6), 5.64189583547474e-07);
    RTEST(result, 1e-13, erfcx(1e7), 5.64189583547753e-08);

    // change of algorithm at 5e7
    RTEST(result, 1e-13, erfcx(5e7), 1.12837916709551e-08);
    RTEST(result, 1e-13, erfcx(5.000000000001e7), 1.12837916709529e-08);

    RTEST(result, 1e-13, erfcx(1e8), 5.64189583547756e-09);
    RTEST(result, 1e-13, erfcx(1e20), 5.64189583547756e-21);
    RTEST(result, 1e-13, erfcx(1e40), 5.64189583547756e-41);
    RTEST(result, 1e-13, erfcx(1e80), 5.64189583547756e-81);
    RTEST(result, 1e-13, erfcx(1e120), 5.64189583547756e-121);
    RTEST(result, 1e-13, erfcx(1e160), 5.64189583547756e-161);

    /***************   Dawson   *****************/

    RTEST(result, 1e-13, dawson(  0.0e+00),    0.000000000000000e+00);

    // rough logarithmic grid
    RTEST(result, 1e-13, dawson(-1.0e-275),  -1.000000000000000e-275);
    RTEST(result, 1e-13, dawson( 1.0e-275),   1.000000000000000e-275);
    RTEST(result, 1e-13, dawson(-1.0e-225),  -1.000000000000000e-225);
    RTEST(result, 1e-13, dawson( 1.0e-225),   1.000000000000000e-225);
    RTEST(result, 1e-13, dawson(-1.0e-175),  -1.000000000000000e-175);
    RTEST(result, 1e-13, dawson( 1.0e-175),   1.000000000000000e-175);
    RTEST(result, 1e-13, dawson(-1.0e-125),  -1.000000000000000e-125);
    RTEST(result, 1e-13, dawson( 1.0e-125),   1.000000000000000e-125);
    RTEST(result, 1e-13, dawson( -1.0e-75),   -1.000000000000000e-75);
    RTEST(result, 1e-13, dawson(  1.0e-75),    1.000000000000000e-75);
    RTEST(result, 1e-13, dawson( -1.0e-25),   -1.000000000000000e-25);
    RTEST(result, 1e-13, dawson(  1.0e-25),    1.000000000000000e-25);
    RTEST(result, 1e-13, dawson( -1.0e+25),   -5.000000000000000e-26);
    RTEST(result, 1e-13, dawson(  1.0e+25),    5.000000000000000e-26);
    RTEST(result, 1e-13, dawson( -1.0e+75),   -5.000000000000001e-76);
    RTEST(result, 1e-13, dawson(  1.0e+75),    5.000000000000001e-76);
    RTEST(result, 1e-13, dawson(-1.0e+125),  -5.000000000000000e-126);
    RTEST(result, 1e-13, dawson( 1.0e+125),   5.000000000000000e-126);
    RTEST(result, 1e-13, dawson(-1.0e+175),  -5.000000000000001e-176);
    RTEST(result, 1e-13, dawson( 1.0e+175),   5.000000000000001e-176);
    RTEST(result, 1e-13, dawson(-1.0e+225),  -5.000000000000001e-226);
    RTEST(result, 1e-13, dawson( 1.0e+225),   5.000000000000001e-226);
    RTEST(result, 1e-13, dawson(-1.0e+275),  -5.000000000000000e-276);
    RTEST(result, 1e-13, dawson( 1.0e+275),   5.000000000000000e-276);

    // medium logarithmic grid
    RTEST(result, 1e-13, dawson( -1.0e-15),   -1.000000000000000e-15);
    RTEST(result, 1e-13, dawson(  1.0e-15),    1.000000000000000e-15);
    RTEST(result, 1e-13, dawson( -1.0e-13),   -1.000000000000000e-13);
    RTEST(result, 1e-13, dawson(  1.0e-13),    1.000000000000000e-13);
    RTEST(result, 1e-13, dawson( -1.0e-11),   -9.999999999999999e-12);
    RTEST(result, 1e-13, dawson(  1.0e-11),    9.999999999999999e-12);
    RTEST(result, 1e-13, dawson( -1.0e-09),   -1.000000000000000e-09);
    RTEST(result, 1e-13, dawson(  1.0e-09),    1.000000000000000e-09);
    RTEST(result, 1e-13, dawson( -1.0e-07),   -9.999999999999933e-08);
    RTEST(result, 1e-13, dawson(  1.0e-07),    9.999999999999933e-08);
    RTEST(result, 1e-13, dawson( -1.0e-05),   -9.999999999333333e-06);
    RTEST(result, 1e-13, dawson(  1.0e-05),    9.999999999333333e-06);
    RTEST(result, 1e-13, dawson( -1.0e-03),   -9.999993333336000e-04);
    RTEST(result, 1e-13, dawson(  1.0e-03),    9.999993333336000e-04);
    RTEST(result, 1e-13, dawson( -1.0e-01),   -9.933599239785290e-02);
    RTEST(result, 1e-13, dawson(  1.0e-01),    9.933599239785290e-02);
    RTEST(result, 1e-13, dawson( -1.0e+01),   -5.025384718759854e-02);
    RTEST(result, 1e-13, dawson(  1.0e+01),    5.025384718759854e-02);
    RTEST(result, 1e-13, dawson( -1.0e+03),   -5.000002500003750e-04);
    RTEST(result, 1e-13, dawson(  1.0e+03),    5.000002500003750e-04);
    RTEST(result, 1e-13, dawson( -1.0e+05),   -5.000000000250000e-06);
    RTEST(result, 1e-13, dawson(  1.0e+05),    5.000000000250000e-06);
    RTEST(result, 1e-13, dawson( -1.0e+07),   -5.000000000000026e-08);
    RTEST(result, 1e-13, dawson(  1.0e+07),    5.000000000000026e-08);
    RTEST(result, 1e-13, dawson( -1.0e+09),   -5.000000000000000e-10);
    RTEST(result, 1e-13, dawson(  1.0e+09),    5.000000000000000e-10);
    RTEST(result, 1e-13, dawson( -1.0e+11),   -5.000000000000000e-12);
    RTEST(result, 1e-13, dawson(  1.0e+11),    5.000000000000000e-12);
    RTEST(result, 1e-13, dawson( -1.0e+13),   -5.000000000000000e-14);
    RTEST(result, 1e-13, dawson(  1.0e+13),    5.000000000000000e-14);
    RTEST(result, 1e-13, dawson( -1.0e+15),   -5.000000000000000e-16);
    RTEST(result, 1e-13, dawson(  1.0e+15),    5.000000000000000e-16);

    // fine logarithmic grid
    RTEST(result, 1e-13, dawson( -1.0e-03),   -9.999993333336000e-04);
    RTEST(result, 1e-13, dawson(  1.0e-03),    9.999993333336000e-04);
    RTEST(result, 1e-13, dawson( -1.5e-03),   -1.499997750002025e-03);
    RTEST(result, 1e-13, dawson(  1.5e-03),    1.499997750002025e-03);
    RTEST(result, 1e-13, dawson( -2.2e-03),   -2.199992901347076e-03);
    RTEST(result, 1e-13, dawson(  2.2e-03),    2.199992901347076e-03);
    RTEST(result, 1e-13, dawson( -3.3e-03),   -3.299976042104361e-03);
    RTEST(result, 1e-13, dawson(  3.3e-03),    3.299976042104361e-03);
    RTEST(result, 1e-13, dawson( -4.7e-03),   -4.699930785278250e-03);
    RTEST(result, 1e-13, dawson(  4.7e-03),    4.699930785278250e-03);
    RTEST(result, 1e-13, dawson( -6.8e-03),   -6.799790382543772e-03);
    RTEST(result, 1e-13, dawson(  6.8e-03),    6.799790382543772e-03);
    RTEST(result, 1e-13, dawson( -1.0e-02),   -9.999333359999240e-03);
    RTEST(result, 1e-13, dawson(  1.0e-02),    9.999333359999240e-03);
    RTEST(result, 1e-13, dawson( -1.5e-02),   -1.499775020248698e-02);
    RTEST(result, 1e-13, dawson(  1.5e-02),    1.499775020248698e-02);
    RTEST(result, 1e-13, dawson( -2.2e-02),   -2.199290270744517e-02);
    RTEST(result, 1e-13, dawson(  2.2e-02),    2.199290270744517e-02);
    RTEST(result, 1e-13, dawson( -3.3e-02),   -3.297605243285839e-02);
    RTEST(result, 1e-13, dawson(  3.3e-02),    3.297605243285839e-02);
    RTEST(result, 1e-13, dawson( -4.7e-02),   -4.693084578675417e-02);
    RTEST(result, 1e-13, dawson(  4.7e-02),    4.693084578675417e-02);
    RTEST(result, 1e-13, dawson( -6.8e-02),   -6.779076587058307e-02);
    RTEST(result, 1e-13, dawson(  6.8e-02),    6.779076587058307e-02);
    RTEST(result, 1e-13, dawson( -1.0e-01),   -9.933599239785290e-02);
    RTEST(result, 1e-13, dawson(  1.0e-01),    9.933599239785290e-02);
    RTEST(result, 1e-13, dawson( -1.5e-01),   -1.477701204696679e-01);
    RTEST(result, 1e-13, dawson(  1.5e-01),    1.477701204696679e-01);
    RTEST(result, 1e-13, dawson( -2.2e-01),   -2.130368833188383e-01);
    RTEST(result, 1e-13, dawson(  2.2e-01),    2.130368833188383e-01);
    RTEST(result, 1e-13, dawson( -3.3e-01),   -3.070539097862983e-01);
    RTEST(result, 1e-13, dawson(  3.3e-01),    3.070539097862983e-01);
    RTEST(result, 1e-13, dawson( -4.7e-01),   -4.065327472834942e-01);
    RTEST(result, 1e-13, dawson(  4.7e-01),    4.065327472834942e-01);
    RTEST(result, 1e-13, dawson( -6.8e-01),   -5.045130066149965e-01);
    RTEST(result, 1e-13, dawson(  6.8e-01),    5.045130066149965e-01);
    RTEST(result, 1e-13, dawson( -1.0e+00),   -5.380795069127684e-01);
    RTEST(result, 1e-13, dawson(  1.0e+00),    5.380795069127684e-01);
    RTEST(result, 1e-13, dawson( -1.5e+00),   -4.282490710853987e-01);
    RTEST(result, 1e-13, dawson(  1.5e+00),    4.282490710853987e-01);
    RTEST(result, 1e-13, dawson( -2.2e+00),   -2.645107599508320e-01);
    RTEST(result, 1e-13, dawson(  2.2e+00),    2.645107599508320e-01);
    RTEST(result, 1e-13, dawson( -3.3e+00),   -1.597885804744951e-01);
    RTEST(result, 1e-13, dawson(  3.3e+00),    1.597885804744951e-01);
    RTEST(result, 1e-13, dawson( -4.7e+00),   -1.089766845891902e-01);
    RTEST(result, 1e-13, dawson(  4.7e+00),    1.089766845891902e-01);
    RTEST(result, 1e-13, dawson( -6.8e+00),   -7.435180005364808e-02);
    RTEST(result, 1e-13, dawson(  6.8e+00),    7.435180005364808e-02);
    RTEST(result, 1e-13, dawson( -1.0e+01),   -5.025384718759854e-02);
    RTEST(result, 1e-13, dawson(  1.0e+01),    5.025384718759854e-02);
    RTEST(result, 1e-13, dawson( -1.5e+01),   -3.340790680863923e-02);
    RTEST(result, 1e-13, dawson(  1.5e+01),    3.340790680863923e-02);
    RTEST(result, 1e-13, dawson( -2.2e+01),   -2.275082445761012e-02);
    RTEST(result, 1e-13, dawson(  2.2e+01),    2.275082445761012e-02);
    RTEST(result, 1e-13, dawson( -3.3e+01),   -1.515848137422974e-02);
    RTEST(result, 1e-13, dawson(  3.3e+01),    1.515848137422974e-02);
    RTEST(result, 1e-13, dawson( -4.7e+01),   -1.064070745357484e-02);
    RTEST(result, 1e-13, dawson(  4.7e+01),    1.064070745357484e-02);
    RTEST(result, 1e-13, dawson( -6.8e+01),   -7.353736519000915e-03);
    RTEST(result, 1e-13, dawson(  6.8e+01),    7.353736519000915e-03);
    RTEST(result, 1e-13, dawson( -1.0e+02),   -5.000250037509379e-03);
    RTEST(result, 1e-13, dawson(  1.0e+02),    5.000250037509379e-03);
    RTEST(result, 1e-13, dawson( -1.5e+02),   -3.333407412346228e-03);
    RTEST(result, 1e-13, dawson(  1.5e+02),    3.333407412346228e-03);
    RTEST(result, 1e-13, dawson( -2.2e+02),   -2.272750752042480e-03);
    RTEST(result, 1e-13, dawson(  2.2e+02),    2.272750752042480e-03);
    RTEST(result, 1e-13, dawson( -3.3e+02),   -1.515158471865865e-03);
    RTEST(result, 1e-13, dawson(  3.3e+02),    1.515158471865865e-03);
    RTEST(result, 1e-13, dawson( -4.7e+02),   -1.063832195194684e-03);
    RTEST(result, 1e-13, dawson(  4.7e+02),    1.063832195194684e-03);
    RTEST(result, 1e-13, dawson( -6.8e+02),   -7.352949127341079e-04);
    RTEST(result, 1e-13, dawson(  6.8e+02),    7.352949127341079e-04);
    RTEST(result, 1e-13, dawson( -1.0e+03),   -5.000002500003750e-04);
    RTEST(result, 1e-13, dawson(  1.0e+03),    5.000002500003750e-04);
    RTEST(result, 1e-13, dawson( -1.5e+03),   -3.333334074074568e-04);
    RTEST(result, 1e-13, dawson(  1.5e+03),    3.333334074074568e-04);
    RTEST(result, 1e-13, dawson( -2.2e+03),   -2.272727507513221e-04);
    RTEST(result, 1e-13, dawson(  2.2e+03),    2.272727507513221e-04);
    RTEST(result, 1e-13, dawson( -3.3e+03),   -1.515151584717710e-04);
    RTEST(result, 1e-13, dawson(  3.3e+03),    1.515151584717710e-04);
    RTEST(result, 1e-13, dawson( -4.7e+03),   -1.063829811313487e-04);
    RTEST(result, 1e-13, dawson(  4.7e+03),    1.063829811313487e-04);
    RTEST(result, 1e-13, dawson( -6.8e+03),   -7.352941255979039e-05);
    RTEST(result, 1e-13, dawson(  6.8e+03),    7.352941255979039e-05);

    /***************   im_w_of_x   *****************/

    RTEST(result, 1e-13, im_w_of_x(  0.0e+00),   0.000000000000000e+00);

    // rough logarithmic grid
    RTEST(result, 1e-13, im_w_of_x(-1.0e-275), -1.128379167095513e-275);
    RTEST(result, 1e-13, im_w_of_x( 1.0e-275),  1.128379167095513e-275);
    RTEST(result, 1e-13, im_w_of_x(-1.0e-225), -1.128379167095512e-225);
    RTEST(result, 1e-13, im_w_of_x( 1.0e-225),  1.128379167095512e-225);
    RTEST(result, 1e-13, im_w_of_x(-1.0e-175), -1.128379167095513e-175);
    RTEST(result, 1e-13, im_w_of_x( 1.0e-175),  1.128379167095513e-175);
    RTEST(result, 1e-13, im_w_of_x(-1.0e-125), -1.128379167095513e-125);
    RTEST(result, 1e-13, im_w_of_x( 1.0e-125),  1.128379167095513e-125);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-75),  -1.128379167095512e-75);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-75),   1.128379167095512e-75);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-25),  -1.128379167095513e-25);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-25),   1.128379167095513e-25);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+25),  -5.641895835477563e-26);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+25),   5.641895835477563e-26);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+75),  -5.641895835477564e-76);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+75),   5.641895835477564e-76);
    RTEST(result, 1e-13, im_w_of_x(-1.0e+125), -5.641895835477563e-126);
    RTEST(result, 1e-13, im_w_of_x( 1.0e+125),  5.641895835477563e-126);
    RTEST(result, 1e-13, im_w_of_x(-1.0e+175), -5.641895835477563e-176);
    RTEST(result, 1e-13, im_w_of_x( 1.0e+175),  5.641895835477563e-176);
    RTEST(result, 1e-13, im_w_of_x(-1.0e+225), -5.641895835477563e-226);
    RTEST(result, 1e-13, im_w_of_x( 1.0e+225),  5.641895835477563e-226);
    RTEST(result, 1e-13, im_w_of_x(-1.0e+275), -5.641895835477563e-276);
    RTEST(result, 1e-13, im_w_of_x( 1.0e+275),  5.641895835477563e-276);

    // medium logarithmic grid
    RTEST(result, 1e-13, im_w_of_x( -1.0e-15),  -1.128379167095513e-15);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-15),   1.128379167095513e-15);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-13),  -1.128379167095513e-13);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-13),   1.128379167095513e-13);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-11),  -1.128379167095512e-11);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-11),   1.128379167095512e-11);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-09),  -1.128379167095513e-09);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-09),   1.128379167095513e-09);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-07),  -1.128379167095505e-07);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-07),   1.128379167095505e-07);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-05),  -1.128379167020287e-05);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-05),   1.128379167020287e-05);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-03),  -1.128378414843035e-03);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-03),   1.128378414843035e-03);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-01),  -1.120886643644954e-01);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-01),   1.120886643644954e-01);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+01),  -5.670539423288760e-02);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+01),   5.670539423288760e-02);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+03),  -5.641898656429711e-04);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+03),   5.641898656429711e-04);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+05),  -5.641895835759658e-06);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+05),   5.641895835759658e-06);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+07),  -5.641895835477592e-08);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+07),   5.641895835477592e-08);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+09),  -5.641895835477563e-10);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+09),   5.641895835477563e-10);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+11),  -5.641895835477562e-12);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+11),   5.641895835477562e-12);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+13),  -5.641895835477563e-14);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+13),   5.641895835477563e-14);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+15),  -5.641895835477563e-16);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+15),   5.641895835477563e-16);

    // fine logarithmic grid
    RTEST(result, 1e-13, im_w_of_x( -1.0e-03),  -1.128378414843035e-03);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-03),   1.128378414843035e-03);
    RTEST(result, 1e-13, im_w_of_x( -1.5e-03),  -1.692566211792428e-03);
    RTEST(result, 1e-13, im_w_of_x(  1.5e-03),   1.692566211792428e-03);
    RTEST(result, 1e-13, im_w_of_x( -2.2e-03),  -2.482426157638054e-03);
    RTEST(result, 1e-13, im_w_of_x(  2.2e-03),   2.482426157638054e-03);
    RTEST(result, 1e-13, im_w_of_x( -3.3e-03),  -3.723624217824865e-03);
    RTEST(result, 1e-13, im_w_of_x(  3.3e-03),   3.723624217824865e-03);
    RTEST(result, 1e-13, im_w_of_x( -4.7e-03),  -5.303303984898830e-03);
    RTEST(result, 1e-13, im_w_of_x(  4.7e-03),   5.303303984898830e-03);
    RTEST(result, 1e-13, im_w_of_x( -6.8e-03),  -7.672741808278817e-03);
    RTEST(result, 1e-13, im_w_of_x(  6.8e-03),   7.672741808278817e-03);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-02),  -1.128303944826631e-02);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-02),   1.128303944826631e-02);
    RTEST(result, 1e-13, im_w_of_x( -1.5e-02),  -1.692314888178881e-02);
    RTEST(result, 1e-13, im_w_of_x(  1.5e-02),   1.692314888178881e-02);
    RTEST(result, 1e-13, im_w_of_x( -2.2e-02),  -2.481633323903963e-02);
    RTEST(result, 1e-13, im_w_of_x(  2.2e-02),   2.481633323903963e-02);
    RTEST(result, 1e-13, im_w_of_x( -3.3e-02),  -3.720949057828670e-02);
    RTEST(result, 1e-13, im_w_of_x(  3.3e-02),   3.720949057828670e-02);
    RTEST(result, 1e-13, im_w_of_x( -4.7e-02),  -5.295578867994562e-02);
    RTEST(result, 1e-13, im_w_of_x(  4.7e-02),   5.295578867994562e-02);
    RTEST(result, 1e-13, im_w_of_x( -6.8e-02),  -7.649368792981542e-02);
    RTEST(result, 1e-13, im_w_of_x(  6.8e-02),   7.649368792981542e-02);
    RTEST(result, 1e-13, im_w_of_x( -1.0e-01),  -1.120886643644954e-01);
    RTEST(result, 1e-13, im_w_of_x(  1.0e-01),   1.120886643644954e-01);
    RTEST(result, 1e-13, im_w_of_x( -1.5e-01),  -1.667407254571674e-01);
    RTEST(result, 1e-13, im_w_of_x(  1.5e-01),   1.667407254571674e-01);
    RTEST(result, 1e-13, im_w_of_x( -2.2e-01),  -2.403863809599346e-01);
    RTEST(result, 1e-13, im_w_of_x(  2.2e-01),   2.403863809599346e-01);
    RTEST(result, 1e-13, im_w_of_x( -3.3e-01),  -3.464732349780839e-01);
    RTEST(result, 1e-13, im_w_of_x(  3.3e-01),   3.464732349780839e-01);
    RTEST(result, 1e-13, im_w_of_x( -4.7e-01),  -4.587230827767997e-01);
    RTEST(result, 1e-13, im_w_of_x(  4.7e-01),   4.587230827767997e-01);
    RTEST(result, 1e-13, im_w_of_x( -6.8e-01),  -5.692819661930826e-01);
    RTEST(result, 1e-13, im_w_of_x(  6.8e-01),   5.692819661930826e-01);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+00),  -6.071577058413937e-01);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+00),   6.071577058413937e-01);
    RTEST(result, 1e-13, im_w_of_x( -1.5e+00),  -4.832273301407691e-01);
    RTEST(result, 1e-13, im_w_of_x(  1.5e+00),   4.832273301407691e-01);
    RTEST(result, 1e-13, im_w_of_x( -2.2e+00),  -2.984684310011209e-01);
    RTEST(result, 1e-13, im_w_of_x(  2.2e+00),   2.984684310011209e-01);
    RTEST(result, 1e-13, im_w_of_x( -3.3e+00),  -1.803021053471850e-01);
    RTEST(result, 1e-13, im_w_of_x(  3.3e+00),   1.803021053471850e-01);
    RTEST(result, 1e-13, im_w_of_x( -4.7e+00),  -1.229670205895808e-01);
    RTEST(result, 1e-13, im_w_of_x(  4.7e+00),   1.229670205895808e-01);
    RTEST(result, 1e-13, im_w_of_x( -6.8e+00),  -8.389702221658751e-02);
    RTEST(result, 1e-13, im_w_of_x(  6.8e+00),   8.389702221658751e-02);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+01),  -5.670539423288760e-02);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+01),   5.670539423288760e-02);
    RTEST(result, 1e-13, im_w_of_x( -1.5e+01),  -3.769678605913683e-02);
    RTEST(result, 1e-13, im_w_of_x(  1.5e+01),   3.769678605913683e-02);
    RTEST(result, 1e-13, im_w_of_x( -2.2e+01),  -2.567155635221433e-02);
    RTEST(result, 1e-13, im_w_of_x(  2.2e+01),   2.567155635221433e-02);
    RTEST(result, 1e-13, im_w_of_x( -3.3e+01),  -1.710451458748619e-02);
    RTEST(result, 1e-13, im_w_of_x(  3.3e+01),   1.710451458748619e-02);
    RTEST(result, 1e-13, im_w_of_x( -4.7e+01),  -1.200675261377179e-02);
    RTEST(result, 1e-13, im_w_of_x(  4.7e+01),   1.200675261377179e-02);
    RTEST(result, 1e-13, im_w_of_x( -6.8e+01),  -8.297803088350106e-03);
    RTEST(result, 1e-13, im_w_of_x(  6.8e+01),   8.297803088350106e-03);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+02),  -5.642177972594138e-03);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+02),   5.642177972594138e-03);
    RTEST(result, 1e-13, im_w_of_x( -1.5e+02),  -3.761347479533245e-03);
    RTEST(result, 1e-13, im_w_of_x(  1.5e+02),   3.761347479533245e-03);
    RTEST(result, 1e-13, im_w_of_x( -2.2e+02),  -2.564524600605394e-03);
    RTEST(result, 1e-13, im_w_of_x(  2.2e+02),   2.564524600605394e-03);
    RTEST(result, 1e-13, im_w_of_x( -3.3e+02),  -1.709673254501715e-03);
    RTEST(result, 1e-13, im_w_of_x(  3.3e+02),   1.709673254501715e-03);
    RTEST(result, 1e-13, im_w_of_x( -4.7e+02),  -1.200406086343168e-03);
    RTEST(result, 1e-13, im_w_of_x(  4.7e+02),   1.200406086343168e-03);
    RTEST(result, 1e-13, im_w_of_x( -6.8e+02),  -8.296914612004802e-04);
    RTEST(result, 1e-13, im_w_of_x(  6.8e+02),   8.296914612004802e-04);
    RTEST(result, 1e-13, im_w_of_x( -1.0e+03),  -5.641898656429711e-04);
    RTEST(result, 1e-13, im_w_of_x(  1.0e+03),   5.641898656429711e-04);
    RTEST(result, 1e-13, im_w_of_x( -1.5e+03),  -3.761264726155353e-04);
    RTEST(result, 1e-13, im_w_of_x(  1.5e+03),   3.761264726155353e-04);
    RTEST(result, 1e-13, im_w_of_x( -2.2e+03),  -2.564498371962829e-04);
    RTEST(result, 1e-13, im_w_of_x(  2.2e+03),   2.564498371962829e-04);
    RTEST(result, 1e-13, im_w_of_x( -3.3e+03),  -1.709665483187216e-04);
    RTEST(result, 1e-13, im_w_of_x(  3.3e+03),   1.709665483187216e-04);
    RTEST(result, 1e-13, im_w_of_x( -4.7e+03),  -1.200403396421289e-04);
    RTEST(result, 1e-13, im_w_of_x(  4.7e+03),   1.200403396421289e-04);
    RTEST(result, 1e-13, im_w_of_x( -6.8e+03),  -8.296905730123859e-05);
    RTEST(result, 1e-13, im_w_of_x(  6.8e+03),   8.296905730123859e-05);

    // integer steps for 100/(1+x) to test each Chebychev polynomial

    // i=0
    RTEST(result, 1e-13, im_w_of_x(   9.999999999999990e+14),    5.641895835477568e-16);
    RTEST(result, 1e-13, im_w_of_x(   1.990000000000000e+02),    2.835159332882219e-03);
    RTEST(result, 1e-13, im_w_of_x(   9.900000000001000e+01),    5.699175456562523e-03);
    // i=1
    RTEST(result, 1e-13, im_w_of_x(   9.899999999999001e+01),    5.699175456563675e-03);
    RTEST(result, 1e-13, im_w_of_x(   6.566666666666667e+01),    8.592716124394034e-03);
    RTEST(result, 1e-13, im_w_of_x(   4.900000000000250e+01),    1.151647239933575e-02);
    // i=2
    RTEST(result, 1e-13, im_w_of_x(   4.899999999999750e+01),    1.151647239933692e-02);
    RTEST(result, 1e-13, im_w_of_x(   3.900000000000000e+01),    1.447115983118770e-02);
    RTEST(result, 1e-13, im_w_of_x(   3.233333333333444e+01),    1.745751971255112e-02);
    // i=3
    RTEST(result, 1e-13, im_w_of_x(   3.233333333333222e+01),    1.745751971255237e-02);
    RTEST(result, 1e-13, im_w_of_x(   2.757142857142857e+01),    2.047632042032461e-02);
    RTEST(result, 1e-13, im_w_of_x(   2.400000000000063e+01),    2.352835885019649e-02);
    // i=4
    RTEST(result, 1e-13, im_w_of_x(   2.399999999999937e+01),    2.352835885019778e-02);
    RTEST(result, 1e-13, im_w_of_x(   2.122222222222222e+01),    2.661446195248900e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.900000000000040e+01),    2.973548837855889e-02);
    // i=5
    RTEST(result, 1e-13, im_w_of_x(   1.899999999999960e+01),    2.973548837856019e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.718181818181818e+01),    3.289233024809358e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.566666666666694e+01),    3.608591504876176e-02);
    // i=6
    RTEST(result, 1e-13, im_w_of_x(   1.566666666666639e+01),    3.608591504876305e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.438461538461539e+01),    3.931720768113434e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.328571428571449e+01),    4.258721266328377e-02);
    // i=7
    RTEST(result, 1e-13, im_w_of_x(   1.328571428571408e+01),    4.258721266328509e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.233333333333333e+01),    4.589697651136774e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.150000000000016e+01),    4.924759031451532e-02);
    // i=8
    RTEST(result, 1e-13, im_w_of_x(   1.149999999999985e+01),    4.924759031451667e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.076470588235294e+01),    5.264019252484897e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.011111111111123e+01),    5.607597198622275e-02);
    // i=9
    RTEST(result, 1e-13, im_w_of_x(   1.011111111111099e+01),    5.607597198622413e-02);
    RTEST(result, 1e-13, im_w_of_x(   9.526315789473685e+00),    5.955617122865677e-02);
    RTEST(result, 1e-13, im_w_of_x(   9.000000000000099e+00),    6.308209005925758e-02);
    // i=10
    RTEST(result, 1e-13, im_w_of_x(   8.999999999999901e+00),    6.308209005925899e-02);
    RTEST(result, 1e-13, im_w_of_x(   8.523809523809524e+00),    6.665508948510822e-02);
    RTEST(result, 1e-13, im_w_of_x(   8.090909090909173e+00),    7.027659600899987e-02);
    // i=11
    RTEST(result, 1e-13, im_w_of_x(   8.090909090909008e+00),    7.027659600900131e-02);
    RTEST(result, 1e-13, im_w_of_x(   7.695652173913043e+00),    7.394810634551917e-02);
    RTEST(result, 1e-13, im_w_of_x(   7.333333333333403e+00),    7.767119261280968e-02);
    // i=12
    RTEST(result, 1e-13, im_w_of_x(   7.333333333333265e+00),    7.767119261281118e-02);
    RTEST(result, 1e-13, im_w_of_x(   7.000000000000000e+00),    8.144750806500296e-02);
    RTEST(result, 1e-13, im_w_of_x(   6.692307692307751e+00),    8.527879344196090e-02);
    // i=13
    RTEST(result, 1e-13, im_w_of_x(   6.692307692307633e+00),    8.527879344196243e-02);
    RTEST(result, 1e-13, im_w_of_x(   6.407407407407407e+00),    8.916688402758272e-02);
    RTEST(result, 1e-13, im_w_of_x(   6.142857142857194e+00),    9.311371752599210e-02);
    // i=14
    RTEST(result, 1e-13, im_w_of_x(   6.142857142857092e+00),    9.311371752599369e-02);
    RTEST(result, 1e-13, im_w_of_x(   5.896551724137931e+00),    9.712134288803234e-02);
    RTEST(result, 1e-13, im_w_of_x(   5.666666666666710e+00),    1.011919302500480e-01);
    // i=15
    RTEST(result, 1e-13, im_w_of_x(   5.666666666666623e+00),    1.011919302500496e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.451612903225806e+00),    1.053277821860331e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.250000000000039e+00),    1.095313465268615e-01);
    // i=16
    RTEST(result, 1e-13, im_w_of_x(   5.249999999999961e+00),    1.095313465268632e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.060606060606061e+00),    1.138052310742711e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.882352941176505e+00),    1.181522206443761e-01);
    // i=17
    RTEST(result, 1e-13, im_w_of_x(   4.882352941176436e+00),    1.181522206443779e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.714285714285714e+00),    1.225752970344747e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.555555555555586e+00),    1.270776627388220e-01);
    // i=18
    RTEST(result, 1e-13, im_w_of_x(   4.555555555555525e+00),    1.270776627388239e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.405405405405405e+00),    1.316627695565670e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.263157894736870e+00),    1.363343536160989e-01);
    // i=19
    RTEST(result, 1e-13, im_w_of_x(   4.263157894736815e+00),    1.363343536161008e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.128205128205129e+00),    1.410964786980336e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.000000000000025e+00),    1.459535899001518e-01);
    // i=20
    RTEST(result, 1e-13, im_w_of_x(   3.999999999999975e+00),    1.459535899001538e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.878048780487805e+00),    1.509105794054894e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.761904761904784e+00),    1.559728651294047e-01);
    // i=21
    RTEST(result, 1e-13, im_w_of_x(   3.761904761904740e+00),    1.559728651294068e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.651162790697675e+00),    1.611464811601701e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.545454545454566e+00),    1.664381761867625e-01);
    // i=22
    RTEST(result, 1e-13, im_w_of_x(   3.545454545454525e+00),    1.664381761867646e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.444444444444445e+00),    1.718555127968045e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.347826086956540e+00),    1.774069571354460e-01);
    // i=23
    RTEST(result, 1e-13, im_w_of_x(   3.347826086956503e+00),    1.774069571354482e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.255319148936170e+00),    1.831019455981526e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.166666666666684e+00),    1.889509136398863e-01);
    // i=24
    RTEST(result, 1e-13, im_w_of_x(   3.166666666666649e+00),    1.889509136398887e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.081632653061225e+00),    1.949652719154663e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.000000000000016e+00),    2.011573170375991e-01);
    // i=25
    RTEST(result, 1e-13, im_w_of_x(   2.999999999999984e+00),    2.011573170376016e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.921568627450980e+00),    2.075400681396657e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.846153846153861e+00),    2.141270257454428e-01);
    // i=26
    RTEST(result, 1e-13, im_w_of_x(   2.846153846153832e+00),    2.141270257454454e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.773584905660377e+00),    2.209318555484517e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.703703703703717e+00),    2.279680058639590e-01);
    // i=27
    RTEST(result, 1e-13, im_w_of_x(   2.703703703703690e+00),    2.279680058639618e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.636363636363636e+00),    2.352482730405781e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.571428571428584e+00),    2.427843334378600e-01);
    // i=28
    RTEST(result, 1e-13, im_w_of_x(   2.571428571428559e+00),    2.427843334378630e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.508771929824561e+00),    2.505862633181274e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.448275862068977e+00),    2.586620690076943e-01);
    // i=29
    RTEST(result, 1e-13, im_w_of_x(   2.448275862068954e+00),    2.586620690076976e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.389830508474576e+00),    2.670172490028069e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.333333333333345e+00),    2.756544075416439e-01);
    // i=30
    RTEST(result, 1e-13, im_w_of_x(   2.333333333333322e+00),    2.756544075416475e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.278688524590164e+00),    2.845729358625366e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.225806451612914e+00),    2.937687733030435e-01);
    // i=31
    RTEST(result, 1e-13, im_w_of_x(   2.225806451612893e+00),    2.937687733030472e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.174603174603174e+00),    3.032342559561739e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.125000000000010e+00),    3.129580561460697e-01);
    // i=32
    RTEST(result, 1e-13, im_w_of_x(   2.124999999999990e+00),    3.129580561460736e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.076923076923077e+00),    3.229252118151739e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.030303030303040e+00),    3.331172412518210e-01);
    // i=33
    RTEST(result, 1e-13, im_w_of_x(   2.030303030303021e+00),    3.331172412518251e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.985074626865672e+00),    3.435123355791175e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.941176470588244e+00),    3.540856191459628e-01);
    // i=34
    RTEST(result, 1e-13, im_w_of_x(   1.941176470588227e+00),    3.540856191459670e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.898550724637681e+00),    3.648094664214367e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.857142857142865e+00),    3.756538631552732e-01);
    // i=35
    RTEST(result, 1e-13, im_w_of_x(   1.857142857142849e+00),    3.756538631552775e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.816901408450704e+00),    3.865867993569494e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.777777777777786e+00),    3.975746819667473e-01);
    // i=36
    RTEST(result, 1e-13, im_w_of_x(   1.777777777777770e+00),    3.975746819667516e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.739726027397260e+00),    4.085827558380871e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.702702702702710e+00),    4.195755227119879e-01);
    // i=37
    RTEST(result, 1e-13, im_w_of_x(   1.702702702702696e+00),    4.195755227119923e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.666666666666667e+00),    4.305171491400668e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.631578947368428e+00),    4.413718557086579e-01);
    // i=38
    RTEST(result, 1e-13, im_w_of_x(   1.631578947368414e+00),    4.413718557086622e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.597402597402597e+00),    4.521042813555960e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.564102564102571e+00),    4.626798179874742e-01);
    // i=39
    RTEST(result, 1e-13, im_w_of_x(   1.564102564102558e+00),    4.626798179874784e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.531645569620253e+00),    4.730649119500522e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.500000000000006e+00),    4.832273301407671e-01);
    // i=40
    RTEST(result, 1e-13, im_w_of_x(   1.499999999999994e+00),    4.832273301407711e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.469135802469136e+00),    4.931363896571986e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.439024390243908e+00),    5.027631508344628e-01);
    // i=41
    RTEST(result, 1e-13, im_w_of_x(   1.439024390243897e+00),    5.027631508344665e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.409638554216868e+00),    5.120805743341599e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.380952380952387e+00),    5.210636436087347e-01);
    // i=42
    RTEST(result, 1e-13, im_w_of_x(   1.380952380952375e+00),    5.210636436087382e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.352941176470588e+00),    5.296894545860749e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.325581395348843e+00),    5.379372748096367e-01);
    // i=43
    RTEST(result, 1e-13, im_w_of_x(   1.325581395348832e+00),    5.379372748096400e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.298850574712644e+00),    5.457885745433008e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.272727272727278e+00),    5.532270325208557e-01);
    // i=44
    RTEST(result, 1e-13, im_w_of_x(   1.272727272727268e+00),    5.532270325208585e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.247191011235955e+00),    5.602385191029849e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.222222222222227e+00),    5.668110596133782e-01);
    // i=45
    RTEST(result, 1e-13, im_w_of_x(   1.222222222222217e+00),    5.668110596133806e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.197802197802198e+00),    5.729347805745570e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.173913043478266e+00),    5.786018414649849e-01);
    // i=46
    RTEST(result, 1e-13, im_w_of_x(   1.173913043478256e+00),    5.786018414649872e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.150537634408602e+00),    5.838063544840782e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.127659574468090e+00),    5.885442946501848e-01);
    // i=47
    RTEST(result, 1e-13, im_w_of_x(   1.127659574468081e+00),    5.885442946501865e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.105263157894737e+00),    5.928134023776949e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.083333333333337e+00),    5.966130804899524e-01);
    // i=48
    RTEST(result, 1e-13, im_w_of_x(   1.083333333333329e+00),    5.966130804899538e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.061855670103093e+00),    5.999442874311427e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.040816326530616e+00),    6.028094282472860e-01);
    // i=49
    RTEST(result, 1e-13, im_w_of_x(   1.040816326530608e+00),    6.028094282472871e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.020202020202020e+00),    6.052122447181988e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.000000000000004e+00),    6.071577058413935e-01);
    // i=50
    RTEST(result, 1e-13, im_w_of_x(   9.999999999999960e-01),    6.071577058413941e-01);
    RTEST(result, 1e-13, im_w_of_x(   9.801980198019802e-01),    6.086518996979112e-01);
    RTEST(result, 1e-13, im_w_of_x(   9.607843137254941e-01),    6.097019275702407e-01);
    // i=51
    RTEST(result, 1e-13, im_w_of_x(   9.607843137254863e-01),    6.097019275702411e-01);
    RTEST(result, 1e-13, im_w_of_x(   9.417475728155340e-01),    6.103158010349921e-01);
    RTEST(result, 1e-13, im_w_of_x(   9.230769230769267e-01),    6.105023426181485e-01);
    // i=52
    RTEST(result, 1e-13, im_w_of_x(   9.230769230769194e-01),    6.105023426181484e-01);
    RTEST(result, 1e-13, im_w_of_x(   9.047619047619047e-01),    6.102710904787984e-01);
    RTEST(result, 1e-13, im_w_of_x(   8.867924528301923e-01),    6.096322074779641e-01);
    // i=53
    RTEST(result, 1e-13, im_w_of_x(   8.867924528301852e-01),    6.096322074779638e-01);
    RTEST(result, 1e-13, im_w_of_x(   8.691588785046729e-01),    6.085963948921743e-01);
    RTEST(result, 1e-13, im_w_of_x(   8.518518518518552e-01),    6.071748109461655e-01);
    // i=54
    RTEST(result, 1e-13, im_w_of_x(   8.518518518518485e-01),    6.071748109461649e-01);
    RTEST(result, 1e-13, im_w_of_x(   8.348623853211010e-01),    6.053789942648607e-01);
    RTEST(result, 1e-13, im_w_of_x(   8.181818181818215e-01),    6.032207922808449e-01);
    // i=55
    RTEST(result, 1e-13, im_w_of_x(   8.181818181818148e-01),    6.032207922808439e-01);
    RTEST(result, 1e-13, im_w_of_x(   8.018018018018018e-01),    6.007122945790411e-01);
    RTEST(result, 1e-13, im_w_of_x(   7.857142857142889e-01),    5.978657711144963e-01);
    // i=56
    RTEST(result, 1e-13, im_w_of_x(   7.857142857142825e-01),    5.978657711144952e-01);
    RTEST(result, 1e-13, im_w_of_x(   7.699115044247788e-01),    5.946936152011272e-01);
    RTEST(result, 1e-13, im_w_of_x(   7.543859649122837e-01),    5.912082911383517e-01);
    // i=57
    RTEST(result, 1e-13, im_w_of_x(   7.543859649122777e-01),    5.912082911383502e-01);
    RTEST(result, 1e-13, im_w_of_x(   7.391304347826086e-01),    5.874222863177538e-01);
    RTEST(result, 1e-13, im_w_of_x(   7.241379310344858e-01),    5.833480676328135e-01);
    // i=58
    RTEST(result, 1e-13, im_w_of_x(   7.241379310344798e-01),    5.833480676328119e-01);
    RTEST(result, 1e-13, im_w_of_x(   7.094017094017093e-01),    5.789980420003302e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.949152542372909e-01),    5.743845207921575e-01);
    // i=59
    RTEST(result, 1e-13, im_w_of_x(   6.949152542372852e-01),    5.743845207921554e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.806722689075630e-01),    5.695196879693124e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.666666666666694e-01),    5.644155717072944e-01);
    // i=60
    RTEST(result, 1e-13, im_w_of_x(   6.666666666666639e-01),    5.644155717072924e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.528925619834711e-01),    5.590840193006392e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.393442622950847e-01),    5.535366751363738e-01);
    // i=61
    RTEST(result, 1e-13, im_w_of_x(   6.393442622950793e-01),    5.535366751363715e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.260162601626016e-01),    5.477849615292567e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.129032258064542e-01),    5.418400622166584e-01);
    // i=62
    RTEST(result, 1e-13, im_w_of_x(   6.129032258064491e-01),    5.418400622166561e-01);
    RTEST(result, 1e-13, im_w_of_x(   6.000000000000001e-01),    5.357129083168283e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.873015873015899e-01),    5.294141665613977e-01);
    // i=63
    RTEST(result, 1e-13, im_w_of_x(   5.873015873015848e-01),    5.294141665613952e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.748031496062993e-01),    5.229542296204844e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.625000000000024e-01),    5.163432083470821e-01);
    // i=64
    RTEST(result, 1e-13, im_w_of_x(   5.624999999999976e-01),    5.163432083470796e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.503875968992249e-01),    5.095909257757789e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.384615384615408e-01),    5.027069127197041e-01);
    // i=65
    RTEST(result, 1e-13, im_w_of_x(   5.384615384615361e-01),    5.027069127197012e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.267175572519085e-01),    4.957004048182317e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.151515151515174e-01),    4.885803408968826e-01);
    // i=66
    RTEST(result, 1e-13, im_w_of_x(   5.151515151515129e-01),    4.885803408968797e-01);
    RTEST(result, 1e-13, im_w_of_x(   5.037593984962405e-01),    4.813553625093524e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.925373134328381e-01),    4.740338145402393e-01);
    // i=67
    RTEST(result, 1e-13, im_w_of_x(   4.925373134328337e-01),    4.740338145402364e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.814814814814814e-01),    4.666237467551144e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.705882352941197e-01),    4.591329161927341e-01);
    // i=68
    RTEST(result, 1e-13, im_w_of_x(   4.705882352941155e-01),    4.591329161927311e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.598540145985401e-01),    4.515687903016827e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.492753623188426e-01),    4.439385507312684e-01);
    // i=69
    RTEST(result, 1e-13, im_w_of_x(   4.492753623188386e-01),    4.439385507312654e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.388489208633093e-01),    4.362490976933090e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.285714285714306e-01),    4.285070548182904e-01);
    // i=70
    RTEST(result, 1e-13, im_w_of_x(   4.285714285714266e-01),    4.285070548182874e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.184397163120568e-01),    4.207187744354848e-01);
    RTEST(result, 1e-13, im_w_of_x(   4.084507042253540e-01),    4.128903432127477e-01);
    // i=71
    RTEST(result, 1e-13, im_w_of_x(   4.084507042253502e-01),    4.128903432127445e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.986013986013985e-01),    4.050275880971084e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.888888888888908e-01),    3.971360825027044e-01);
    // i=72
    RTEST(result, 1e-13, im_w_of_x(   3.888888888888871e-01),    3.971360825027013e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.793103448275863e-01),    3.892211526973144e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.698630136986321e-01),    3.812878843435226e-01);
    // i=73
    RTEST(result, 1e-13, im_w_of_x(   3.698630136986283e-01),    3.812878843435195e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.605442176870748e-01),    3.733411291546030e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.513513513513531e-01),    3.653855116293745e-01);
    // i=74
    RTEST(result, 1e-13, im_w_of_x(   3.513513513513495e-01),    3.653855116293714e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.422818791946309e-01),    3.574254358337423e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.333333333333350e-01),    3.494650922002370e-01);
    // i=75
    RTEST(result, 1e-13, im_w_of_x(   3.333333333333315e-01),    3.494650922002338e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.245033112582782e-01),    3.415084643197962e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.157894736842122e-01),    3.335593357031167e-01);
    // i=76
    RTEST(result, 1e-13, im_w_of_x(   3.157894736842088e-01),    3.335593357031135e-01);
    RTEST(result, 1e-13, im_w_of_x(   3.071895424836601e-01),    3.256212964913635e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.987012987013005e-01),    3.176977500986562e-01);
    // i=77
    RTEST(result, 1e-13, im_w_of_x(   2.987012987012969e-01),    3.176977500986530e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.903225806451613e-01),    3.097919197707839e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.820512820512837e-01),    3.019068550468424e-01);
    // i=78
    RTEST(result, 1e-13, im_w_of_x(   2.820512820512804e-01),    3.019068550468392e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.738853503184713e-01),    2.940454381121446e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.658227848101282e-01),    2.862103900326554e-01);
    // i=79
    RTEST(result, 1e-13, im_w_of_x(   2.658227848101249e-01),    2.862103900326522e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.578616352201257e-01),    2.784042768625366e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.500000000000016e-01),    2.706295156179890e-01);
    // i=80
    RTEST(result, 1e-13, im_w_of_x(   2.499999999999984e-01),    2.706295156179859e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.422360248447204e-01),    2.628883801116380e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.345679012345694e-01),    2.551830066430521e-01);
    // i=81
    RTEST(result, 1e-13, im_w_of_x(   2.345679012345663e-01),    2.551830066430490e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.269938650306749e-01),    2.475153995418103e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.195121951219527e-01),    2.398874365606541e-01);
    // i=82
    RTEST(result, 1e-13, im_w_of_x(   2.195121951219496e-01),    2.398874365606511e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.121212121212122e-01),    2.323008741168892e-01);
    RTEST(result, 1e-13, im_w_of_x(   2.048192771084352e-01),    2.247573523811336e-01);
    // i=83
    RTEST(result, 1e-13, im_w_of_x(   2.048192771084323e-01),    2.247573523811306e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.976047904191616e-01),    2.172584002129732e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.904761904761918e-01),    2.098054399438566e-01);
    // i=84
    RTEST(result, 1e-13, im_w_of_x(   1.904761904761891e-01),    2.098054399438536e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.834319526627219e-01),    2.023997920078819e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.764705882352955e-01),    1.950426794217791e-01);
    // i=85
    RTEST(result, 1e-13, im_w_of_x(   1.764705882352928e-01),    1.950426794217762e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.695906432748537e-01),    1.877352321155810e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.627906976744200e-01),    1.804784911160209e-01);
    // i=86
    RTEST(result, 1e-13, im_w_of_x(   1.627906976744173e-01),    1.804784911160180e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.560693641618498e-01),    1.732734125847965e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.494252873563231e-01),    1.661208717142802e-01);
    // i=87
    RTEST(result, 1e-13, im_w_of_x(   1.494252873563204e-01),    1.661208717142774e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.428571428571428e-01),    1.590216664832867e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.363636363636376e-01),    1.519765212758721e-01);
    // i=88
    RTEST(result, 1e-13, im_w_of_x(   1.363636363636351e-01),    1.519765212758692e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.299435028248588e-01),    1.449860903661028e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.235955056179787e-01),    1.380509612720384e-01);
    // i=89
    RTEST(result, 1e-13, im_w_of_x(   1.235955056179763e-01),    1.380509612720356e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.173184357541899e-01),    1.311716579820805e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.111111111111123e-01),    1.243486440570980e-01);
    // i=90
    RTEST(result, 1e-13, im_w_of_x(   1.111111111111098e-01),    1.243486440570953e-01);
    RTEST(result, 1e-13, im_w_of_x(   1.049723756906078e-01),    1.175823256116063e-01);
    RTEST(result, 1e-13, im_w_of_x(   9.890109890110010e-02),    1.108730541774922e-01);
    // i=91
    RTEST(result, 1e-13, im_w_of_x(   9.890109890109766e-02),    1.108730541774896e-01);
    RTEST(result, 1e-13, im_w_of_x(   9.289617486338808e-02),    1.042211294536169e-01);
    RTEST(result, 1e-13, im_w_of_x(   8.695652173913171e-02),    9.762680194480361e-02);
    // i=92
    RTEST(result, 1e-13, im_w_of_x(   8.695652173912927e-02),    9.762680194480083e-02);
    RTEST(result, 1e-13, im_w_of_x(   8.108108108108114e-02),    9.109027549354108e-02);
    RTEST(result, 1e-13, im_w_of_x(   7.526881720430212e-02),    8.461170970787571e-02);
    // i=93
    RTEST(result, 1e-13, im_w_of_x(   7.526881720429990e-02),    8.461170970787335e-02);
    RTEST(result, 1e-13, im_w_of_x(   6.951871657754016e-02),    7.819122228877137e-02);
    RTEST(result, 1e-13, im_w_of_x(   6.382978723404364e-02),    7.182889126034092e-02);
    // i=94
    RTEST(result, 1e-13, im_w_of_x(   6.382978723404142e-02),    7.182889126033844e-02);
    RTEST(result, 1e-13, im_w_of_x(   5.820105820105814e-02),    6.552475710614740e-02);
    RTEST(result, 1e-13, im_w_of_x(   5.263157894736947e-02),    5.927882481487420e-02);
    // i=95
    RTEST(result, 1e-13, im_w_of_x(   5.263157894736725e-02),    5.927882481487175e-02);
    RTEST(result, 1e-13, im_w_of_x(   4.712041884816753e-02),    5.309106583845361e-02);
    RTEST(result, 1e-13, im_w_of_x(   4.166666666666785e-02),    4.696141996583530e-02);
    // i=96
    RTEST(result, 1e-13, im_w_of_x(   4.166666666666563e-02),    4.696141996583272e-02);
    RTEST(result, 1e-13, im_w_of_x(   3.626943005181338e-02),    4.088979711535257e-02);
    RTEST(result, 1e-13, im_w_of_x(   3.092783505154739e-02),    3.487607904873747e-02);
    // i=97
    RTEST(result, 1e-13, im_w_of_x(   3.092783505154539e-02),    3.487607904873510e-02);
    RTEST(result, 1e-13, im_w_of_x(   2.564102564102555e-02),    2.892012100959479e-02);
    RTEST(result, 1e-13, im_w_of_x(   2.040816326530726e-02),    2.302175328924806e-02);
    // i=98
    RTEST(result, 1e-13, im_w_of_x(   2.040816326530503e-02),    2.302175328924556e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.522842639593902e-02),    1.718078272261780e-02);
    RTEST(result, 1e-13, im_w_of_x(   1.010101010101105e-02),    1.139699411689208e-02);
    // i=99
    RTEST(result, 1e-13, im_w_of_x(   1.010101010100906e-02),    1.139699411688982e-02);
    RTEST(result, 1e-13, im_w_of_x(   5.025125628140614e-03),    5.670151615501411e-03);
    RTEST(result, 1e-13, im_w_of_x(   8.881784197001252e-16),    1.002202025453436e-15);
    // i=100
    RTEST(result, 1e-13, im_w_of_x(  -9.992007221626409e-16),   -1.127477278635115e-15);
    RTEST(result, 1e-13, im_w_of_x(  -4.975124378109430e-03),   -5.613734067773474e-03);
    RTEST(result, 1e-13, im_w_of_x(  -9.900990099008911e-03),   -1.117134086083436e-02);
    // i=101
    RTEST(result, 1e-13, im_w_of_x(  -9.900990099010909e-03),   -1.117134086083661e-02);
    RTEST(result, 1e-13, im_w_of_x(  -1.477832512315269e-02),   -1.667312645723201e-02);
    RTEST(result, 1e-13, im_w_of_x(  -1.960784313725394e-02),   -2.211941166301962e-02);

    /***************   terminate   *****************/

    printf("%i/%i tests failed\n", result.failed, result.total);
    return result.failed;
}
