// Copyright (c) 2006-2018 Maxim Khizhinsky
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying
// file LICENSE or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef CDSUNIT_QUEUE_TEST_BOUNDED_QUEUE_H
#define CDSUNIT_QUEUE_TEST_BOUNDED_QUEUE_H

#include <cds_test/check_size.h>

namespace cds_test {

    class bounded_queue : public ::testing::Test
    {
    protected:
        template <typename Queue>
        void test( Queue& q )
        {
            typedef typename Queue::value_type value_type;
            value_type it;

            const size_t nSize = q.capacity();

            ASSERT_TRUE( q.empty());
            ASSERT_CONTAINER_SIZE( q, 0 );

            // enqueue/dequeue
            for ( unsigned pass = 0; pass < 3; ++pass ) {
                for ( size_t i = 0; i < nSize; ++i ) {
                    it = static_cast<value_type>( i );
                    ASSERT_TRUE( q.enqueue( it ));
                    ASSERT_CONTAINER_SIZE( q, i + 1 );
                }
                ASSERT_FALSE( q.empty());
                ASSERT_CONTAINER_SIZE( q, nSize );
                ASSERT_FALSE( q.enqueue( static_cast<value_type>( nSize ) * 2 ));

                for ( size_t i = 0; i < nSize; ++i ) {
                    it = -1;
                    ASSERT_TRUE( q.dequeue( it ));
                    ASSERT_EQ( it, static_cast<value_type>( i ));
                    ASSERT_CONTAINER_SIZE( q, nSize - i - 1 );
                }
                ASSERT_TRUE( q.empty());
                ASSERT_CONTAINER_SIZE( q, 0 );
            }

            // push/pop
            for ( unsigned pass = 0; pass < 3; ++pass ) {
                for ( size_t i = 0; i < nSize; ++i ) {
                    it = static_cast<value_type>( i );
                    ASSERT_TRUE( q.push( it ));
                    ASSERT_CONTAINER_SIZE( q, i + 1 );
                }
                ASSERT_FALSE( q.empty());
                ASSERT_CONTAINER_SIZE( q, nSize );

                for ( size_t i = 0; i < nSize; ++i ) {
                    it = -1;
                    ASSERT_TRUE( q.pop( it ));
                    ASSERT_EQ( it, static_cast<value_type>( i ));
                    ASSERT_CONTAINER_SIZE( q, nSize - i - 1 );
                }
                ASSERT_TRUE( q.empty());
                ASSERT_CONTAINER_SIZE( q, 0 );
            }

            // push/pop with lambda
            for ( unsigned pass = 0; pass < 3; ++pass ) {
                for ( size_t i = 0; i < nSize; ++i ) {
                    it = static_cast<value_type>( i );
                    ASSERT_NE( it, -1 );
                    auto f = [&it]( value_type& dest ) { dest = it; it = -1; };
                    if ( i & 1 )
                        ASSERT_TRUE( q.enqueue_with( f ));
                    else
                        ASSERT_TRUE( q.push_with( f ));
                    ASSERT_EQ( it, -1 );
                    ASSERT_CONTAINER_SIZE( q, i + 1 );
                }
                ASSERT_FALSE( q.empty());
                ASSERT_CONTAINER_SIZE( q, nSize );

                for ( size_t i = 0; i < nSize; ++i ) {
                    it = -1;
                    auto f = [&it]( value_type& src ) { it = src; src = -1; };
                    if ( i & 1 )
                        ASSERT_TRUE( q.pop_with( f ));
                    else
                        ASSERT_TRUE( q.dequeue_with( f ));
                    ASSERT_EQ( it, static_cast<value_type>( i ));
                    ASSERT_CONTAINER_SIZE( q, nSize - i - 1 );
                }
                ASSERT_TRUE( q.empty());
                ASSERT_CONTAINER_SIZE( q, 0u );
            }

            for ( size_t i = 0; i < nSize; ++i ) {
                ASSERT_TRUE( q.push( static_cast<value_type>(i)));
            }
            ASSERT_FALSE( q.empty());
            ASSERT_CONTAINER_SIZE( q, nSize );

            // push in full queue
            ASSERT_FALSE( q.push( static_cast<int>(nSize * 2 )));
            ASSERT_FALSE( q.empty());
            ASSERT_CONTAINER_SIZE( q, nSize );
            it = static_cast<int>( nSize * 2 );
            ASSERT_FALSE( q.enqueue( it ));
            ASSERT_FALSE( q.empty());
            ASSERT_CONTAINER_SIZE( q, nSize );

            // clear
            q.clear();
            ASSERT_TRUE( q.empty());
            ASSERT_CONTAINER_SIZE( q, 0u );

            // pop from empty queue
            it = static_cast<int>(nSize * 2);
            ASSERT_FALSE( q.pop( it ));
            ASSERT_EQ( it, static_cast<value_type>( nSize * 2 ));
            ASSERT_TRUE( q.empty());
            ASSERT_CONTAINER_SIZE( q, 0u );

            ASSERT_FALSE( q.dequeue( it ));
            ASSERT_EQ( it, static_cast<value_type>( nSize * 2 ));
            ASSERT_TRUE( q.empty());
            ASSERT_CONTAINER_SIZE( q, 0u );
        }

        template <class Queue>
        void test_string( Queue& q )
        {
            std::string str[3];
            str[0] = "one";
            str[1] = "two";
            str[2] = "three";
            const size_t nSize = sizeof( str ) / sizeof( str[0] );

            // emplace
            for ( size_t i = 0; i < nSize; ++i ) {
                ASSERT_TRUE( q.emplace( str[i].c_str()));
                ASSERT_CONTAINER_SIZE( q, i + 1 );
            }
            ASSERT_FALSE( q.empty());
            ASSERT_CONTAINER_SIZE( q, nSize );

            {
                std::string s;
                auto f = [&s]( std::string& src ) {
                    ASSERT_FALSE( src.empty());
                    s = std::move( src );
                    ASSERT_NE( s, src );
                };
                for ( size_t i = 0; i < nSize; ++i ) {
                    if ( i & 1 )
                        ASSERT_TRUE( q.pop_with( f ));
                    else
                        ASSERT_TRUE( q.dequeue_with( f ));

                    ASSERT_CONTAINER_SIZE( q, nSize - i - 1 );
                    ASSERT_EQ( s, str[i] );
                }
            }
            ASSERT_TRUE( q.empty());
            ASSERT_CONTAINER_SIZE( q, 0 );


            // move push
            for ( size_t i = 0; i < nSize; ++i ) {
                std::string s = str[i];
                ASSERT_FALSE( s.empty());
                if ( i & 1 )
                    ASSERT_TRUE( q.enqueue( std::move( s )));
                else
                    ASSERT_TRUE( q.push( std::move( s )));
                ASSERT_TRUE( s.empty());
                ASSERT_CONTAINER_SIZE( q, i + 1 );
            }
            ASSERT_FALSE( q.empty());
            ASSERT_CONTAINER_SIZE( q, nSize );

            for ( size_t i = 0; i < nSize; ++i ) {
                std::string s;
                ASSERT_TRUE( q.pop( s ));
                ASSERT_CONTAINER_SIZE( q, nSize - i - 1 );
                ASSERT_EQ( s, str[i] );
            }
            ASSERT_TRUE( q.empty());
            ASSERT_CONTAINER_SIZE( q, 0 );
        }

    };

} // namespace cds_test

#endif // CDSUNIT_QUEUE_TEST_BOUNDED_QUEUE_H
