/*****************************************************************************
 Copyright (c) 2018-2020, Intel Corporation

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:

     * Redistributions of source code must retain the above copyright notice,
       this list of conditions and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
     * Neither the name of Intel Corporation nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*****************************************************************************/

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include <intel-ipsec-mb.h>

#include "gcm_ctr_vectors_test.h"
#include "utils.h"

int aes_test(struct IMB_MGR *mb_mgr);

struct aes_vector {
	const uint8_t *K;          /* key */
	const uint8_t *IV;         /* initialization vector */
	const uint8_t *P;          /* plain text */
	uint64_t       Plen;       /* plain text length */
        const uint8_t *C;          /* cipher text - same length as plain text */
        uint32_t       Klen;       /* key length */
};

/* =================================================================== */
/* =================================================================== */

/*
 *  AES Test vectors from
 *  http://csrc.nist.gov/publications/fips/fips197/fips-197.pdf
 */

/*  128-bit */
static const uint8_t K1[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t IV1[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t P1[] = {
        0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
        0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
        0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
        0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
        0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
        0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
        0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
        0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
};
static const uint8_t C1[] = {
        0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46,
        0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d,
        0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee,
        0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2,
        0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b,
        0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16,
        0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac, 0x09,
        0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7
};

/*  192-bit */
static const uint8_t K2[] = {
        0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
        0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
        0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b
};
static const uint8_t IV2[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t P2[] = {
        0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
        0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
        0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
        0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
        0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
        0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
        0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
        0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
};
static const uint8_t C2[] = {
        0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d,
        0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8,
        0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4,
        0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a,
        0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0,
        0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0,
        0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81,
        0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd
};

/*  256-bit */
static const uint8_t K3[] = {
        0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
        0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
        0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
        0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4
};
static const uint8_t IV3[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t P3[] = {
        0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
        0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
        0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
        0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
        0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
        0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
        0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
        0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
};
static const uint8_t C3[] = {
        0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba,
        0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6,
        0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d,
        0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d,
        0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf,
        0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61,
        0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc,
        0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b
};

/* Extra AES test vectors */

/*  128-bit */
static const uint8_t K4[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t IV4[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t P4[] = {
        0xf7, 0xcd, 0x12, 0xfb, 0x4f, 0x8e, 0x50, 0xab,
        0x35, 0x8e, 0x56, 0xf9, 0x83, 0x53, 0x9a, 0x1a,
        0xfc, 0x47, 0x3c, 0x96, 0x01, 0xfe, 0x01, 0x87,
        0xd5, 0xde, 0x46, 0x24, 0x5c, 0x62, 0x8f, 0xba,
        0xba, 0x91, 0x17, 0x8d, 0xba, 0x5a, 0x79, 0xb1,
        0x57, 0x05, 0x4d, 0x08, 0xba, 0x1f, 0x30, 0xd3,
        0x80, 0x40, 0xe9, 0x37, 0xb0, 0xd6, 0x34, 0x87,
        0x33, 0xdd, 0xc0, 0x5b, 0x2d, 0x58, 0x1d, 0x2a,
        0x7b, 0xb6, 0xe3, 0xd0, 0xc8, 0xa0, 0x7a, 0x69,
        0xc8, 0x5d, 0x10, 0xa2, 0xc3, 0x39, 0xca, 0xaf,
        0x40, 0xdc, 0xc7, 0xcb, 0xff, 0x18, 0x7d, 0x51,
        0x06, 0x28, 0x28, 0x1f, 0x3a, 0x9c, 0x18, 0x7d,
        0x5b, 0xb5, 0xe9, 0x20, 0xc2, 0xae, 0x17, 0x7f,
        0xd1, 0x65, 0x7a, 0x75, 0xcf, 0x21, 0xa0, 0x1e,
        0x17, 0x1b, 0xf7, 0xe8, 0x62, 0x5f, 0xaf, 0x34,
        0x7f, 0xd8, 0x18, 0x4a, 0x94, 0xf2, 0x33, 0x90
};
static const uint8_t C4[] = {
        0xf0, 0x8f, 0x91, 0x13, 0x11, 0x01, 0xdc, 0xbb,
        0xcd, 0xf9, 0x95, 0x92, 0xda, 0xbf, 0x2a, 0x86,
        0xea, 0x8d, 0xa6, 0x08, 0xc8, 0xb5, 0x65, 0x82,
        0x93, 0x43, 0xb7, 0x0e, 0x14, 0x36, 0xb4, 0xcf,
        0xd8, 0x11, 0xab, 0x21, 0x5b, 0x64, 0xb8, 0xc5,
        0xee, 0x27, 0x93, 0x66, 0x59, 0xd9, 0x1d, 0xc9,
        0x84, 0x9d, 0x03, 0xbd, 0xab, 0xce, 0x6a, 0x14,
        0x76, 0x73, 0x17, 0xe3, 0xb3, 0xe5, 0x70, 0xe8,
        0xa2, 0xa8, 0xce, 0xb0, 0xf6, 0xc4, 0xc5, 0xb5,
        0x8e, 0x22, 0xef, 0x33, 0xdf, 0x18, 0x42, 0x40,
        0x56, 0xc4, 0xb9, 0x7f, 0x60, 0x9e, 0x8b, 0x45,
        0xc1, 0xbf, 0xa7, 0xfa, 0x1b, 0x3e, 0x02, 0x5d,
        0xb3, 0x04, 0x93, 0x30, 0xf5, 0xff, 0x8e, 0xb6,
        0x0a, 0xfb, 0x41, 0xfe, 0x09, 0xa5, 0x90, 0xc7,
        0x22, 0xab, 0xaa, 0x22, 0x89, 0xd8, 0x3c, 0x4e,
        0x46, 0x18, 0x93, 0xbf, 0x1a, 0xce, 0x77, 0x59
};

/*  192-bit */
static const uint8_t K5[] = {
        0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
        0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
        0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b
};
static const uint8_t IV5[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t P5[] = {
        0x19, 0x08, 0xa3, 0x58, 0x17, 0x14, 0x70, 0x5a,
        0xb8, 0xab, 0x4f, 0x5f, 0xa4, 0x25, 0x2b, 0xec,
        0xb6, 0x74, 0x0b, 0x9d, 0x56, 0x3b, 0xaf, 0xa3,
        0xa4, 0x2d, 0x3e, 0x1f, 0x18, 0x84, 0x3b, 0x4f,
        0x48, 0xd9, 0xa3, 0xfe, 0x59, 0x1e, 0x80, 0x67,
        0x44, 0x35, 0x26, 0x00, 0x78, 0xda, 0x68, 0xfa,
        0x61, 0x9c, 0xd8, 0x8e, 0x5c, 0xc1, 0xff, 0xeb,
        0x9c, 0x7d, 0xe7, 0xa9, 0x38, 0xeb, 0x66, 0xf8,
        0x6a, 0x46, 0x71, 0x51, 0x02, 0xba, 0x8d, 0x70,
        0x55, 0x5b, 0x60, 0xc6, 0x4c, 0xae, 0xda, 0x2e,
        0x17, 0xbb, 0x65, 0xef, 0x60, 0x85, 0x9e, 0x77,
        0xe5, 0x83, 0xef, 0x30, 0x08, 0x3a, 0xba, 0x80,
        0x28, 0xc0, 0xa1, 0x93, 0x4c, 0x2a, 0x0b, 0xe1,
        0xcb, 0xd0, 0xac, 0x72, 0x72, 0x1d, 0x96, 0x76,
        0x0e, 0xc0, 0xec, 0x7d, 0x84, 0xfd, 0xee, 0x08,
        0xa1, 0x11, 0x20, 0x0d, 0x59, 0x5c, 0x06, 0x3f,
        0xa3, 0xf1, 0xd7, 0xa3, 0x1d, 0x29, 0xc3, 0xaa,
        0x05, 0x2b, 0x74, 0x8c, 0x73, 0x60, 0x65, 0x43,
        0x76, 0xd4, 0xd7, 0x7b, 0x5f, 0x40, 0xf4, 0x77,
        0xe1, 0xcc, 0x85, 0x37, 0x1c, 0xd8, 0xda, 0x91,
        0xf0, 0x40, 0xb2, 0x43, 0x2d, 0x87, 0x51, 0xd0,
        0xce, 0x27, 0xa6, 0x60, 0xac, 0x67, 0xea, 0x8b,
        0xae, 0x46, 0x2e, 0x78, 0x06, 0x09, 0x8a, 0x82,
        0xb0, 0x0d, 0x57, 0x56, 0x82, 0xfe, 0x89, 0xd2
};
static const uint8_t C5[] = {
        0xfa, 0x88, 0xb3, 0x4e, 0x7f, 0x3e, 0x78, 0x4d,
        0xfd, 0xb3, 0x38, 0xee, 0xb0, 0xdd, 0x0d, 0xf5,
        0xeb, 0x24, 0xe6, 0x70, 0xd8, 0xac, 0xd7, 0xfa,
        0x41, 0x67, 0x2e, 0x2d, 0x7e, 0x9b, 0x26, 0xac,
        0xf1, 0x0f, 0x1f, 0x47, 0x6d, 0xff, 0x46, 0xd1,
        0x1a, 0xeb, 0xe9, 0x3c, 0x1b, 0x9d, 0x55, 0x86,
        0xde, 0xee, 0x3d, 0xd8, 0x12, 0x05, 0x12, 0x9d,
        0xff, 0x23, 0x97, 0x57, 0xb0, 0xdc, 0x7b, 0x7a,
        0xdf, 0xba, 0x7f, 0x69, 0x85, 0xdf, 0xa9, 0xfd,
        0x3e, 0xa7, 0x36, 0x26, 0x30, 0xdd, 0x07, 0x0f,
        0x89, 0x0b, 0x27, 0x9c, 0x23, 0xa1, 0xfa, 0x7d,
        0x4e, 0x64, 0x50, 0x07, 0x86, 0x13, 0x98, 0xee,
        0x05, 0xc6, 0x6c, 0xd9, 0xd1, 0xe8, 0xb2, 0x6b,
        0xe6, 0x73, 0x06, 0x39, 0xbb, 0x72, 0x74, 0xa3,
        0xc2, 0x1a, 0x40, 0xcd, 0xec, 0x40, 0x8f, 0x44,
        0xf8, 0x86, 0xff, 0x7e, 0xb7, 0xea, 0xda, 0xb0,
        0x5c, 0x25, 0xdf, 0x3f, 0x54, 0xda, 0xca, 0xea,
        0x76, 0xe5, 0xec, 0xbb, 0x21, 0xd3, 0x86, 0x8d,
        0x8a, 0x57, 0xf0, 0x31, 0x9f, 0x56, 0xa3, 0x1b,
        0xf9, 0x55, 0xe6, 0xa6, 0xde, 0xb7, 0x74, 0xcc,
        0x2b, 0x17, 0x9a, 0xe3, 0x1b, 0x74, 0x0d, 0x2b,
        0x99, 0xcd, 0x64, 0xe1, 0x7b, 0x7e, 0x1c, 0xcd,
        0x9b, 0x23, 0x02, 0x7d, 0x86, 0x52, 0xfd, 0x14,
        0x2d, 0xbb, 0x75, 0x3d, 0xa3, 0x3b, 0xc1, 0xe0
};

/*  256-bit */
static const uint8_t K6[] = {
        0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
        0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
        0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
        0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4
};
static const uint8_t IV6[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t P6[] = {
        0x0b, 0xe5, 0x48, 0xa6, 0xa1, 0xbc, 0xac, 0x81,
        0x80, 0x06, 0x5f, 0xae, 0x1e, 0x3f, 0x55, 0x73,
        0x6d, 0x36, 0x7f, 0x57, 0x3d, 0xa4, 0x4a, 0x6b,
        0xb6, 0x65, 0x2f, 0xb7, 0xe8, 0x85, 0x47, 0xe2,
        0x41, 0x42, 0xc2, 0x4e, 0x58, 0xf1, 0xde, 0x42,
        0x9f, 0x15, 0x4c, 0xaf, 0xea, 0x04, 0x20, 0xd0,
        0x1a, 0x19, 0x36, 0x74, 0x71, 0x12, 0x72, 0x1b,
        0xdb, 0x18, 0xf9, 0x0b, 0xb3, 0xf3, 0x63, 0xd4,
        0x62, 0x52, 0x8b, 0x63, 0x0f, 0x6b, 0x4d, 0xb9,
        0x70, 0xd6, 0x91, 0xa0, 0x43, 0x3f, 0x46, 0xfe,
        0x43, 0xbb, 0xb8, 0xdc, 0x5e, 0xdb, 0xd4, 0x1f,
        0xf0, 0x17, 0x94, 0x25, 0xee, 0x55, 0x67, 0xbf,
        0x4d, 0xda, 0x9d, 0xe7, 0x4b, 0xc6, 0x7a, 0xcf,
        0x8f, 0xd7, 0xbb, 0x29, 0x6e, 0x26, 0xd4, 0xc3,
        0x08, 0x9b, 0x67, 0x15, 0xe9, 0x2d, 0x9f, 0x2d,
        0x3c, 0x76, 0x26, 0xd3, 0xda, 0xfe, 0x6e, 0x73,
        0x9d, 0x09, 0x60, 0x4b, 0x35, 0x60, 0xdb, 0x77,
        0xb6, 0xc0, 0x45, 0x91, 0xf9, 0x14, 0x8a, 0x7a,
        0xdd, 0xe2, 0xf1, 0xdf, 0x8f, 0x12, 0x4f, 0xd7,
        0x75, 0xd6, 0x9a, 0x17, 0xda, 0x76, 0x88, 0xf0,
        0xfa, 0x44, 0x27, 0xbe, 0x61, 0xaf, 0x55, 0x9f,
        0xc7, 0xf0, 0x76, 0x77, 0xde, 0xca, 0xd1, 0x47,
        0x51, 0x55, 0xb1, 0xbf, 0xfa, 0x1e, 0xca, 0x28,
        0x17, 0x70, 0xf3, 0xb5, 0xd4, 0x32, 0x47, 0x04,
        0xe0, 0x92, 0xd8, 0xa5, 0x03, 0x69, 0x46, 0x99,
        0x7f, 0x1e, 0x3f, 0xb2, 0x93, 0x36, 0xa3, 0x88,
        0x75, 0x07, 0x68, 0xb8, 0x33, 0xce, 0x17, 0x3f,
        0x5c, 0xb7, 0x1e, 0x93, 0x38, 0xc5, 0x1d, 0x79,
        0x86, 0x7c, 0x9d, 0x9e, 0x2f, 0x69, 0x38, 0x0f,
        0x97, 0x5c, 0x67, 0xbf, 0xa0, 0x8d, 0x37, 0x0b,
        0xd3, 0xb1, 0x04, 0x87, 0x1d, 0x74, 0xfe, 0x30,
        0xfb, 0xd0, 0x22, 0x92, 0xf9, 0xf3, 0x23, 0xc9
};
static const uint8_t C6[] = {
        0x16, 0x60, 0x36, 0xd9, 0xcf, 0xe8, 0xd6, 0x07,
        0x81, 0xdf, 0x28, 0x0a, 0x40, 0x44, 0x61, 0x45,
        0x83, 0x28, 0xd5, 0x1b, 0xf7, 0x55, 0x54, 0x35,
        0xd3, 0x43, 0x73, 0x0e, 0x7a, 0xc3, 0x83, 0xb1,
        0xc9, 0xbd, 0x22, 0x70, 0xf0, 0xde, 0x8f, 0x92,
        0x5e, 0xe1, 0x56, 0xd3, 0x4d, 0x01, 0x64, 0xfa,
        0xe9, 0x83, 0x35, 0x60, 0x80, 0x70, 0xf5, 0xb5,
        0x13, 0x76, 0xd3, 0x88, 0xbb, 0x7f, 0x2d, 0x0a,
        0x31, 0x04, 0xb4, 0x77, 0x47, 0x91, 0x3f, 0xe4,
        0xa9, 0x9a, 0x19, 0xbe, 0xfb, 0xd6, 0x70, 0xae,
        0xb1, 0xea, 0xd5, 0x03, 0xd6, 0xb5, 0xca, 0x76,
        0x5e, 0x0d, 0x21, 0x31, 0x87, 0xf3, 0xb2, 0x2e,
        0xe2, 0xbc, 0x71, 0xb5, 0x8b, 0x7e, 0xa6, 0x09,
        0x78, 0x6e, 0x76, 0xe6, 0x61, 0xdf, 0x86, 0xe6,
        0x8d, 0x2f, 0x12, 0x43, 0x99, 0xf9, 0xf1, 0x86,
        0xf1, 0x55, 0xfd, 0x35, 0xcd, 0xe8, 0x92, 0x4e,
        0x87, 0x33, 0x77, 0x62, 0x64, 0xaa, 0x60, 0x07,
        0x33, 0x08, 0x45, 0xf5, 0xd6, 0xb0, 0x9c, 0xf4,
        0xba, 0xda, 0x17, 0x74, 0x74, 0x23, 0x54, 0x9c,
        0x7e, 0x86, 0x57, 0x83, 0x3d, 0xda, 0xc3, 0xe1,
        0x02, 0x90, 0xe3, 0x69, 0x80, 0x7a, 0x5b, 0x47,
        0xf5, 0xea, 0x83, 0x1a, 0xc6, 0x1a, 0xaa, 0x53,
        0x66, 0xfe, 0xe6, 0xbd, 0x72, 0x9b, 0x8b, 0x96,
        0xdb, 0x94, 0xa9, 0x5b, 0xc3, 0x40, 0x6a, 0xcd,
        0xf4, 0x78, 0x14, 0x29, 0x7b, 0x8f, 0x26, 0xb0,
        0x89, 0xbd, 0x03, 0x55, 0x33, 0x46, 0x4c, 0x96,
        0x2a, 0x58, 0x69, 0x7c, 0x9b, 0xdf, 0xba, 0xb8,
        0x75, 0x5b, 0xbc, 0x4b, 0x19, 0xd3, 0x9d, 0xee,
        0xfd, 0x17, 0x2f, 0x14, 0xea, 0xd9, 0x32, 0xd2,
        0xaa, 0xaf, 0x09, 0xce, 0x81, 0xca, 0x7f, 0xc1,
        0x50, 0x5d, 0x13, 0x3a, 0x91, 0x27, 0x16, 0x97,
        0x57, 0x1f, 0x5d, 0xc5, 0x2e, 0x56, 0xc2, 0xca
};

/*  128-bit */
static const uint8_t K7[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t IV7[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t P7[] = {
        0xdd, 0x14, 0xde, 0x30, 0xe0, 0xfd, 0x7b, 0x2a,
        0x94, 0x8e, 0x28, 0xa0, 0xf6, 0x93, 0x6e, 0xf5,
        0x92, 0x65, 0x1d, 0x5e, 0x78, 0x2a, 0x9d, 0x39,
        0xfc, 0xb8, 0x6d, 0x8b, 0xa5, 0xf4, 0x4b, 0x21,
        0xdd, 0x4e, 0xe9, 0xeb, 0xd7, 0xa7, 0xa1, 0x59,
        0xdc, 0x4c, 0x5e, 0xcc, 0x83, 0xab, 0xd3, 0x45,
        0xfe, 0x2c, 0x73, 0x23, 0xea, 0x45, 0xcb, 0x0c,
        0x12, 0x67, 0x28, 0xcd, 0xef, 0x4e, 0xca, 0xe2,
        0x1d, 0x92, 0x82, 0xd8, 0x0f, 0xa9, 0x36, 0x23,
        0x6d, 0x38, 0x68, 0xac, 0xa0, 0xeb, 0xdc, 0xcc,
        0xdf, 0xb8, 0x3a, 0x53, 0x04, 0x1a, 0x55, 0x27,
        0x8e, 0x22, 0x86, 0x8c, 0xbd, 0xdc, 0x6b, 0x12,
        0x9c, 0x69, 0xd2, 0x7a, 0x4b, 0x52, 0x5d, 0x76,
        0x34, 0xb9, 0x5e, 0x30, 0x0a, 0x8d, 0x1e, 0xf1,
        0x27, 0xda, 0x5b, 0xb9, 0x5e, 0xbf, 0x65, 0x34,
        0x00, 0xb6, 0xd2, 0xb0, 0x89, 0x12, 0xb6, 0x35,
        0xae, 0x27, 0x7f, 0x11, 0xe9, 0xf9, 0x1c, 0x71,
        0xc9, 0x50, 0xfe, 0xd4, 0x76, 0x50, 0x95, 0xf7,
        0xe1, 0x1c, 0x14, 0xcd, 0x67, 0x0f, 0xf0, 0x6d,
        0xa2, 0x93, 0x7b, 0x2c, 0x8d, 0x83, 0x5c, 0xff,
        0xe4, 0x95, 0xf3, 0xa1, 0xfd, 0x00, 0x77, 0x68,
        0x41, 0xb4, 0xfb, 0x81, 0xf4, 0x61, 0x1a, 0x84,
        0x5a, 0x53, 0xc3, 0xdc, 0xba, 0x0d, 0x67, 0x2e,
        0xcf, 0xf2, 0x30, 0xf5, 0x1d, 0xe9, 0xc4, 0x2c,
        0xac, 0x1f, 0xa7, 0x9c, 0x64, 0xfd, 0x45, 0x30,
        0x1b, 0xa1, 0x3b, 0x3d, 0xc7, 0xf5, 0xf9, 0xbb,
        0xba, 0x99, 0xa4, 0x12, 0x6e, 0x4e, 0xea, 0x0b,
        0x29, 0x7f, 0xcd, 0x84, 0x64, 0x50, 0x40, 0xb7,
        0x6a, 0x24, 0x29, 0xa4, 0xa7, 0xa1, 0xef, 0xa9,
        0xcf, 0xdf, 0x09, 0xff, 0xaa, 0x17, 0x5d, 0x82,
        0x74, 0xf5, 0xae, 0xd0, 0xe9, 0xec, 0xad, 0x5e,
        0xa7, 0x84, 0xda, 0xe7, 0x33, 0x58, 0x7e, 0x00,
        0x45, 0x5f, 0xbb, 0x15, 0xa3, 0x65, 0x0e, 0xf5,
        0x7e, 0x27, 0xe7, 0x04, 0x52, 0x58, 0x81, 0xd0,
        0xee, 0x8f, 0xaf, 0xe2, 0x3c, 0xbe, 0x08, 0x97,
        0x8a, 0x97, 0x12, 0xb0, 0x09, 0xfe, 0xa5, 0xeb,
        0xd1, 0x9c, 0x30, 0xe8, 0x9a, 0x3f, 0xe0, 0x38,
        0x34, 0x2b, 0xad, 0xb7, 0xc4, 0xda, 0x54, 0xab,
        0x97, 0x9c, 0x46, 0x2b, 0x2c, 0x0b, 0xb3, 0x49,
        0xcd, 0x9d, 0x32, 0x38, 0x3c, 0x1a, 0x49, 0xdc,
        0x2f, 0xe7, 0xcd, 0x8a, 0xb0, 0x76, 0xcf, 0x30,
        0xea, 0x0b, 0xb0, 0xb7, 0x63, 0xed, 0xb2, 0x8c,
        0xc9, 0x2c, 0xb7, 0x75, 0xa8, 0xf6, 0x63, 0xb6,
        0xcd, 0xb5, 0x63, 0xfb, 0x5f, 0x89, 0xae, 0x3d,
        0x33, 0x73, 0xaf, 0xde, 0xcb, 0x37, 0x0a, 0x50,
        0x6f, 0xae, 0xf3, 0xa6, 0x79, 0x85, 0xdd, 0xc5,
        0x24, 0xc5, 0x29, 0x23, 0x64, 0xef, 0x43, 0xd7,
        0xc4, 0xab, 0xd8, 0xb0, 0x84, 0x26, 0x6b, 0xe8,
        0xb1, 0x5d, 0xb5, 0x69, 0xfb, 0x97, 0x0e, 0x20,
        0xb3, 0xc1, 0x60, 0xad, 0x1a, 0xd2, 0xd6, 0x3a,
        0x73, 0x08, 0xf0, 0x47, 0x5f, 0xcf, 0x15, 0xf7,
        0x7b, 0xf3, 0x69, 0x08, 0x5a, 0x6b, 0x9f, 0xc7,
        0x12, 0xa1, 0xf0, 0xfb, 0x91, 0xc9, 0x07, 0x61,
        0x21, 0xa0, 0x30, 0x4c, 0x16, 0x81, 0xcd, 0x3c,
        0x61, 0xe8, 0x96, 0x91, 0x30, 0xdd, 0x0c, 0x0e,
        0x0b, 0xa1, 0x33, 0x95, 0x67, 0x99, 0xd6, 0x1e,
        0x1a, 0xb3, 0x12, 0xfd, 0xad, 0x46, 0x48, 0x87,
        0x5e, 0xe8, 0xd4, 0xf5, 0xac, 0xdf, 0xa7, 0x37,
        0xb8, 0xa1, 0x62, 0x8c, 0xb8, 0xb6, 0xb0, 0x69,
        0x63, 0x29, 0x60, 0x64, 0x26, 0xc3, 0xf8, 0x18,
        0x8e, 0x46, 0xa0, 0xc5, 0x45, 0x5c, 0x08, 0x2a,
        0xed, 0x29, 0x84, 0x11, 0xea, 0x59, 0xc0, 0x16,
        0xe2, 0x04, 0x30, 0x63, 0x22, 0x87, 0xb6, 0xc7,
        0x81, 0xa6, 0x58, 0xc0, 0xb2, 0xb0, 0x7d, 0xbc,
        0x16, 0x44, 0x6e, 0x5d, 0x6d, 0xce, 0x2a, 0xe0,
        0x20, 0x69, 0x35, 0xa1, 0x5d, 0x17, 0x48, 0x55,
        0x88, 0xfe, 0xde, 0x34, 0xe7, 0x18, 0xbf, 0x7e,
        0x0a, 0x1c, 0x32, 0x88, 0xab, 0xde, 0xe1, 0x02,
        0x61, 0x09, 0x58, 0x96, 0xef, 0x16, 0x73, 0xac,
        0xc0, 0x5c, 0x15, 0xca, 0x9b, 0xea, 0x0e, 0x05,
        0x97, 0x88, 0x09, 0xc5, 0xd0, 0x95, 0x90, 0xae,
        0xa5, 0xb5, 0x28, 0xc6, 0x5a, 0x7b, 0xb3, 0xcc,
        0xae, 0x57, 0x71, 0x83, 0x56, 0x57, 0xca, 0xe8,
        0x8b, 0x21, 0x0c, 0x37, 0x1d, 0xde, 0x85, 0xe2,
        0x1b, 0xa2, 0x38, 0xa0, 0xc5, 0xc7, 0x98, 0x7b,
        0xf9, 0x5e, 0x6a, 0x68, 0xb3, 0xed, 0x49, 0x5e,
        0x46, 0xb9, 0xc9, 0xf6, 0x34, 0xa6, 0x0e, 0xac,
        0x90, 0x72, 0xcf, 0xf8, 0x5b, 0x48, 0x13, 0x40,
        0x7a, 0xce, 0xfd, 0x3c, 0x16, 0xff, 0xb5, 0xea,
        0xb2, 0x56, 0x47, 0xcc, 0x9f, 0xbc, 0xae, 0x4a,
        0xc8, 0xa5, 0x59, 0x57, 0x01, 0xd7, 0x9f, 0xd7,
        0xbf, 0x13, 0xb1, 0xbf, 0xb7, 0x9a, 0xa0, 0xa1,
        0xc6, 0x66, 0x61, 0x96, 0xf2, 0xcd, 0x8c, 0xcb,
        0x3c, 0x67, 0xb5, 0xed, 0xb7, 0xa2, 0x54, 0x84,
        0x3c, 0xcb, 0x7e, 0xb3, 0x97, 0x05, 0xcb, 0x8f,
        0xa9, 0xc6, 0x3c, 0xa2, 0xbd, 0xbf, 0x3a, 0xb8,
        0x92, 0x08, 0x01, 0xea, 0xfd, 0x55, 0x2f, 0x27,
        0x2a, 0x82, 0x38, 0x26, 0x1d, 0x81, 0x19, 0x33,
        0x75, 0x3c, 0xa2, 0x13, 0x1e, 0x58, 0x9f, 0x0b,
        0x08, 0x5d, 0x7a, 0x2c, 0x9a, 0xd1, 0xa5, 0x4c,
        0x41, 0xb4, 0x1d, 0xf8, 0x42, 0x08, 0x87, 0xdd,
        0x8e, 0xc9, 0x05, 0xd2, 0x8c, 0xba, 0x93, 0x28,
        0xbe, 0x4a, 0x14, 0x13, 0x2a, 0x58, 0xf0, 0x1c,
        0xac, 0xc1, 0xc4, 0x49, 0xbc, 0xe1, 0xda, 0xb6,
        0x2d, 0x06, 0x98, 0x32, 0xea, 0xa3, 0x89, 0x11,
        0xca, 0x5f, 0x3e, 0xda, 0x24, 0xe2, 0xdb, 0x1e,
        0xca, 0xf3, 0xc0, 0xc7, 0x64, 0xee, 0x4b, 0x3d,
        0xa2, 0xee, 0x69, 0xb0, 0x3f, 0x2c, 0xd5, 0x49,
        0xba, 0x2d, 0x45, 0x7d, 0xdd, 0xb0, 0x0d, 0xc5,
        0xe0, 0x57, 0x95, 0xbe, 0xf8, 0x4a, 0x11, 0x46,
        0x4c, 0xbb, 0xdf, 0xa8, 0x5a, 0xf9, 0xff, 0x0e,
        0x31, 0xa9, 0x50, 0x5d, 0xc4, 0xb3, 0x3d, 0x09,
        0x46, 0x33, 0x39, 0x31, 0xd5, 0xb3, 0xe5, 0x91,
        0xcf, 0xca, 0x8a, 0xe0, 0xc2, 0x8e, 0xea, 0xbe,
        0x54, 0x64, 0x78, 0x0c, 0x25, 0x1c, 0x17, 0xbc,
        0x49, 0xf9, 0xc0, 0x30, 0x5f, 0x08, 0x04, 0x9d,
        0xb5, 0xe4, 0xeb, 0x9e, 0xe5, 0x1e, 0x6d, 0xbc,
        0x7b, 0xe7, 0xf0, 0xd1, 0xa0, 0x01, 0x18, 0x51,
        0x4f, 0x64, 0xc3, 0x9c, 0x70, 0x25, 0x4f, 0xed,
        0xc7, 0xbc, 0x19, 0x00, 0x09, 0x22, 0x97, 0x5d,
        0x6f, 0xe4, 0x47, 0x98, 0x05, 0xcd, 0xcc, 0xde,
        0xd5, 0xe3, 0xaf, 0xa3, 0xde, 0x69, 0x99, 0x2a,
        0xd1, 0x28, 0x4d, 0x7c, 0x89, 0xa0, 0xdb, 0xae,
        0xf9, 0xf1, 0x4a, 0x46, 0xdf, 0xbe, 0x1d, 0x37,
        0xf2, 0xd5, 0x36, 0x4a, 0x54, 0xe8, 0xc4, 0xfb,
        0x57, 0x77, 0x09, 0x05, 0x31, 0x99, 0xaf, 0x9a,
        0x17, 0xd1, 0x20, 0x93, 0x31, 0x89, 0xff, 0xed,
        0x0f, 0xf8, 0xed, 0xb3, 0xcf, 0x4c, 0x9a, 0x74,
        0xbb, 0x00, 0x36, 0x41, 0xd1, 0x13, 0x68, 0x73,
        0x78, 0x63, 0x42, 0xdd, 0x99, 0x15, 0x9a, 0xf4,
        0xe1, 0xad, 0x6d, 0xf6, 0x5e, 0xca, 0x20, 0x24,
        0xd7, 0x9d, 0x2f, 0x58, 0x97, 0xf7, 0xde, 0x31,
        0x51, 0xa3, 0x1c, 0xe2, 0x66, 0x24, 0x4b, 0xa1,
        0x56, 0x02, 0x32, 0xf4, 0x89, 0xf3, 0x86, 0x9a,
        0x85, 0xda, 0x95, 0xa8, 0x7f, 0x6a, 0x77, 0x02,
        0x3a, 0xba, 0xe0, 0xbe, 0x34, 0x5c, 0x9a, 0x1a
};
static const uint8_t C7[] = {
        0xfb, 0x04, 0xe9, 0x1c, 0xc3, 0x56, 0x9c, 0xb0,
        0xba, 0xc4, 0x66, 0xa3, 0xba, 0x45, 0xac, 0xb8,
        0xd6, 0xd8, 0x95, 0x6c, 0x28, 0xd1, 0x51, 0x6d,
        0xaa, 0x8c, 0x2e, 0xf1, 0x34, 0xab, 0xeb, 0x66,
        0xf9, 0x4e, 0x24, 0x61, 0x1d, 0x16, 0x99, 0xd5,
        0x10, 0x30, 0x42, 0x31, 0x68, 0x98, 0xc5, 0xdb,
        0x0c, 0x9f, 0x0a, 0x1a, 0x65, 0x7d, 0x03, 0x50,
        0xb8, 0x00, 0x0c, 0x40, 0x93, 0x6b, 0xa9, 0x1f,
        0x28, 0x87, 0x01, 0x3c, 0xe9, 0xeb, 0x0e, 0x10,
        0x0f, 0x35, 0xbe, 0x9c, 0x6a, 0xfa, 0x00, 0xac,
        0x25, 0x77, 0x5d, 0x49, 0xde, 0xdc, 0xa1, 0x62,
        0xa7, 0xb7, 0x30, 0x75, 0x36, 0x32, 0x31, 0xab,
        0x40, 0xbb, 0x96, 0xba, 0x46, 0x32, 0x53, 0x8c,
        0x35, 0x7d, 0xa4, 0x21, 0xfa, 0x6a, 0xeb, 0x68,
        0xe4, 0xa4, 0xbf, 0xac, 0x24, 0xbf, 0x59, 0x8e,
        0x98, 0xa6, 0x53, 0xca, 0xe3, 0x69, 0xdd, 0x47,
        0x6e, 0x18, 0x94, 0xf0, 0x40, 0x03, 0x59, 0x93,
        0x96, 0xde, 0x57, 0x96, 0x00, 0xaf, 0x56, 0x88,
        0xb5, 0x0d, 0x55, 0xbc, 0x24, 0xac, 0x11, 0xff,
        0x4d, 0x72, 0x82, 0xda, 0xf2, 0xee, 0xbc, 0x56,
        0x8a, 0x17, 0x24, 0x6b, 0x88, 0x7e, 0x9c, 0xdb,
        0x07, 0xdd, 0xd4, 0x12, 0x15, 0x4d, 0x9e, 0x1a,
        0x57, 0x12, 0x8d, 0x84, 0xdb, 0x17, 0x1a, 0x2f,
        0x7a, 0x3d, 0x4c, 0xbb, 0xc2, 0xb8, 0x73, 0xad,
        0x39, 0x13, 0xf8, 0x2e, 0xfc, 0xf9, 0x3b, 0x64,
        0x06, 0x9e, 0x78, 0x73, 0xff, 0x2b, 0x8c, 0x1b,
        0x4e, 0x21, 0x3e, 0x05, 0x4d, 0xee, 0x9d, 0x39,
        0x7c, 0x61, 0xe1, 0x18, 0x98, 0xe3, 0x50, 0x25,
        0xf9, 0x48, 0x5e, 0x66, 0x9d, 0x41, 0xa2, 0x08,
        0x3f, 0x88, 0x28, 0x03, 0x68, 0x8a, 0xfc, 0xf4,
        0x7a, 0xf5, 0xcb, 0x7d, 0xeb, 0x9e, 0xb2, 0x22,
        0xbc, 0x1a, 0x94, 0x51, 0xa4, 0x7b, 0x9a, 0x2c,
        0xb3, 0x67, 0x60, 0x94, 0x06, 0x31, 0x80, 0xa0,
        0xf7, 0x7f, 0xe8, 0x47, 0x00, 0xab, 0x0b, 0x56,
        0x09, 0xa6, 0xa4, 0x77, 0x18, 0xa5, 0x30, 0x81,
        0xd9, 0x7e, 0x2d, 0x6a, 0x77, 0x34, 0x4e, 0xca,
        0x72, 0x0d, 0xb3, 0x31, 0x87, 0x9c, 0x98, 0xc9,
        0x48, 0x4c, 0xa0, 0x8d, 0xed, 0x9d, 0x7b, 0x9e,
        0xb4, 0xfe, 0x05, 0x7f, 0x93, 0x56, 0xa8, 0x2b,
        0x07, 0x0b, 0xc5, 0x52, 0x96, 0xd5, 0x6a, 0xe4,
        0xf6, 0x38, 0x79, 0x67, 0xd6, 0xfe, 0x8c, 0x0b,
        0x33, 0xe0, 0xe8, 0x15, 0xe7, 0x70, 0x3e, 0xca,
        0xa7, 0x6a, 0xbb, 0x81, 0xf7, 0x94, 0x7f, 0x17,
        0xd6, 0x66, 0x96, 0xbf, 0x1c, 0x8f, 0x71, 0xb6,
        0x9c, 0x5c, 0xe2, 0x61, 0x47, 0x7b, 0x6e, 0xa2,
        0x87, 0x17, 0x55, 0x08, 0x1d, 0x10, 0xb1, 0x34,
        0x3c, 0x21, 0x16, 0x70, 0x3d, 0x0d, 0x93, 0x68,
        0x5e, 0x46, 0x22, 0x45, 0x00, 0xdb, 0xf0, 0x9b,
        0xa1, 0x1f, 0xc7, 0x5b, 0x17, 0xe1, 0x95, 0x07,
        0x57, 0xe5, 0xae, 0x5a, 0x6d, 0x10, 0x83, 0xc4,
        0x1c, 0x0d, 0xf5, 0x73, 0xd3, 0xeb, 0x52, 0x29,
        0x33, 0x4f, 0xb0, 0xe7, 0x5c, 0xf6, 0xdb, 0xb5,
        0x21, 0x6f, 0x35, 0x9a, 0x43, 0x9c, 0x86, 0xeb,
        0x11, 0x95, 0x91, 0x10, 0xa3, 0xbd, 0xe2, 0xe4,
        0x69, 0xac, 0xb1, 0x50, 0xd4, 0xf1, 0x68, 0xe6,
        0x65, 0xb1, 0x96, 0xda, 0xfb, 0xf0, 0x13, 0x06,
        0xa4, 0x63, 0xb6, 0xdb, 0x79, 0x2b, 0x3a, 0xc9,
        0x98, 0x7a, 0x2c, 0x37, 0xf9, 0x4f, 0xa6, 0x93,
        0x9d, 0x3b, 0xb3, 0x06, 0x63, 0xe2, 0xf6, 0x92,
        0x07, 0xe2, 0x82, 0xfd, 0xb5, 0x08, 0x9b, 0x79,
        0x79, 0x78, 0x3b, 0xee, 0x28, 0x54, 0x81, 0x5d,
        0x7a, 0xa3, 0x81, 0x93, 0xa9, 0xc2, 0x59, 0x3f,
        0xb3, 0xc5, 0xcd, 0x89, 0xa2, 0x31, 0xc2, 0xf0,
        0x84, 0x8c, 0x2e, 0x0a, 0xa4, 0x2f, 0x9c, 0xf2,
        0x54, 0x56, 0xec, 0x75, 0x39, 0xd7, 0x92, 0x53,
        0x60, 0x58, 0xf8, 0x81, 0x84, 0x0c, 0x99, 0xc4,
        0x6f, 0x88, 0xf8, 0x6e, 0x6d, 0xd6, 0x08, 0x47,
        0x6a, 0xa4, 0x79, 0xbc, 0xeb, 0x1e, 0x67, 0xd7,
        0xdf, 0x0c, 0x52, 0xdc, 0x74, 0x40, 0x39, 0x17,
        0xdc, 0xd9, 0x13, 0x72, 0x58, 0xc5, 0x30, 0xda,
        0xad, 0x76, 0xa9, 0x9a, 0xad, 0xed, 0xfb, 0x4b,
        0x4e, 0x60, 0xde, 0xc9, 0x18, 0xa0, 0x77, 0x50,
        0x54, 0xfa, 0x00, 0xd6, 0xa9, 0x52, 0xfe, 0x67,
        0x3e, 0xe9, 0xdf, 0x46, 0x14, 0x6c, 0xfb, 0x50,
        0xd6, 0x21, 0xf6, 0xe5, 0xf7, 0x99, 0x38, 0xad,
        0x65, 0xa5, 0x6c, 0x4e, 0x21, 0x31, 0x77, 0x7a,
        0xdc, 0x6f, 0x5d, 0xb5, 0x7f, 0x63, 0xf4, 0xa8,
        0xee, 0x0d, 0x68, 0x10, 0xde, 0x5b, 0x45, 0x4b,
        0x03, 0xd8, 0x55, 0x04, 0x15, 0x6e, 0xc6, 0xb7,
        0xc1, 0x30, 0x29, 0x6a, 0x6c, 0x26, 0xe8, 0x41,
        0x53, 0xb9, 0x82, 0x67, 0x5b, 0xfe, 0xa9, 0x5f,
        0x0b, 0xf8, 0x38, 0xf8, 0xbe, 0x3c, 0x26, 0xf2,
        0x83, 0x94, 0xd6, 0x45, 0x64, 0x1f, 0x17, 0x20,
        0x4d, 0xae, 0x4a, 0x15, 0x27, 0x7d, 0x7f, 0x3b,
        0x71, 0x3c, 0x3a, 0xc3, 0x56, 0x1b, 0xe5, 0xbd,
        0x34, 0x4b, 0x3f, 0x88, 0x3e, 0xcc, 0x98, 0xb5,
        0x5e, 0x8b, 0xab, 0x18, 0x98, 0xf0, 0xef, 0x1b,
        0x78, 0x15, 0xb7, 0x4a, 0x1f, 0xe3, 0x45, 0xc7,
        0x31, 0x34, 0x5a, 0x7b, 0x6e, 0xb8, 0xea, 0xfe,
        0xaf, 0x34, 0x32, 0x45, 0xfa, 0x3e, 0x75, 0x8a,
        0x30, 0x3f, 0xed, 0xe5, 0xfe, 0x66, 0x15, 0xc7,
        0xbe, 0xd9, 0xc7, 0x27, 0x3c, 0x26, 0x66, 0x2d,
        0xa1, 0x0b, 0xb9, 0x1e, 0x17, 0x44, 0xd3, 0x4b,
        0xe6, 0x30, 0x85, 0x9e, 0x29, 0x3d, 0xa9, 0x35,
        0xca, 0x61, 0xea, 0x22, 0x76, 0xdb, 0xce, 0x82,
        0xfe, 0x8b, 0xac, 0xd3, 0x09, 0x90, 0xad, 0xf2,
        0x42, 0x45, 0x8b, 0xbd, 0xad, 0x34, 0x56, 0x67,
        0x3a, 0x81, 0x3d, 0x95, 0x37, 0x72, 0xe6, 0xcc,
        0x20, 0xe7, 0x09, 0x84, 0x99, 0x8b, 0x1a, 0x68,
        0x5f, 0x4e, 0x00, 0x14, 0x3e, 0x94, 0xa7, 0x15,
        0xab, 0xdd, 0x01, 0x2f, 0x9d, 0x57, 0xce, 0x24,
        0x40, 0x97, 0x5e, 0x62, 0x7c, 0x4f, 0xe7, 0x1d,
        0x53, 0x79, 0x05, 0x52, 0x5d, 0xc9, 0xc6, 0xe0,
        0x47, 0xc1, 0xb5, 0x7f, 0x47, 0x28, 0x7d, 0x0b,
        0xa8, 0x51, 0x27, 0xb9, 0x21, 0x97, 0x2d, 0x5b,
        0x03, 0x94, 0x30, 0x63, 0xa5, 0x02, 0x04, 0xf0,
        0x53, 0x53, 0x23, 0xfa, 0x81, 0xd1, 0x31, 0x3b,
        0x63, 0x5d, 0x61, 0x3b, 0x44, 0x19, 0xb2, 0x24,
        0x15, 0x79, 0x54, 0xb0, 0x57, 0x8c, 0x17, 0x0d,
        0x36, 0xad, 0xa3, 0x08, 0x71, 0x60, 0x85, 0xc9,
        0x5e, 0x7b, 0x55, 0x85, 0x8a, 0x90, 0x0b, 0x2c,
        0x2b, 0x9a, 0x5d, 0xb6, 0x0e, 0xb6, 0xa6, 0x1d,
        0xb1, 0xf5, 0xe1, 0xae, 0xf9, 0x94, 0xb6, 0x3d,
        0xd0, 0xad, 0x5b, 0xa7, 0x3a, 0x66, 0xd0, 0x31,
        0x45, 0xcc, 0xb7, 0x7f, 0xce, 0x0f, 0x07, 0x2e,
        0x64, 0x11, 0xe0, 0xcd, 0xac, 0xdb, 0x75, 0xb1,
        0x5a, 0x4a, 0x5b, 0x15, 0x6a, 0xe2, 0x28, 0x8c,
        0x6d, 0xe5, 0x5a, 0x82, 0x62, 0xeb, 0xfc, 0xf5,
        0x9b, 0x67, 0xa0, 0x79, 0x75, 0x24, 0x2e, 0xd4,
        0x3b, 0x53, 0xd4, 0xec, 0x6b, 0x0f, 0x43, 0x22,
        0xe3, 0xc3, 0x75, 0x83, 0x2d, 0x64, 0x5f, 0x8a,
        0x79, 0x49, 0x5f, 0x1a, 0x81, 0xeb, 0xd5, 0x47,
        0xc9, 0xe7, 0xa8, 0x14, 0xd9, 0xcc, 0xb0, 0xa4,
        0xea, 0xfc, 0x12, 0x23, 0xb3, 0x1b, 0x7a, 0xac,
        0x0f, 0x6a, 0x86, 0x4c, 0x4b, 0x91, 0x13, 0xa3,
        0x52, 0x51, 0x69, 0xc8, 0xff, 0x52, 0x8f, 0x44
};

static const struct aes_vector aes_vectors[] = {
        {K1, IV1, P1, sizeof(P1), C1, sizeof(K1)},
        {K2, IV2, P2, sizeof(P2), C2, sizeof(K2)},
        {K3, IV3, P3, sizeof(P3), C3, sizeof(K3)},
        {K4, IV4, P4, sizeof(P4), C4, sizeof(K4)},
        {K5, IV5, P5, sizeof(P5), C5, sizeof(K5)},
        {K6, IV6, P6, sizeof(P6), C6, sizeof(K6)},
        {K7, IV7, P7, sizeof(P7), C7, sizeof(K7)},
};

/* =================================================================== */
/* =================================================================== */

/*  DOCSIS: AES-128 CFB */
static const uint8_t DK1[] = {
        0xe6, 0x60, 0x0f, 0xd8, 0x85, 0x2e, 0xf5, 0xab,
        0xe6, 0x60, 0x0f, 0xd8, 0x85, 0x2e, 0xf5, 0xab
};
static const uint8_t DIV1[] = {
        0x81, 0x0e, 0x52, 0x8e, 0x1c, 0x5f, 0xda, 0x1a,
        0x81, 0x0e, 0x52, 0x8e, 0x1c, 0x5f, 0xda, 0x1a
};
static const uint8_t DP1[] = {
        0x00, 0x01, 0x02, 0x88, 0xee, 0x59, 0x7e
};
static const uint8_t DC1[] = {
        0xfc, 0x68, 0xa3, 0x55, 0x60, 0x37, 0xdc
};

/*  DOCSIS: AES-128 CBC + CFB */
static const uint8_t DK2[] = {
        0xe6, 0x60, 0x0f, 0xd8, 0x85, 0x2e, 0xf5, 0xab,
        0xe6, 0x60, 0x0f, 0xd8, 0x85, 0x2e, 0xf5, 0xab
};
static const uint8_t DIV2[] = {
        0x81, 0x0e, 0x52, 0x8e, 0x1c, 0x5f, 0xda, 0x1a,
        0x81, 0x0e, 0x52, 0x8e, 0x1c, 0x5f, 0xda, 0x1a
};
static const uint8_t DP2[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x91,
        0xd2, 0xd1, 0x9f
};
static const uint8_t DC2[] = {
        0x9d, 0xd1, 0x67, 0x4b, 0xba, 0x61, 0x10, 0x1b,
        0x56, 0x75, 0x64, 0x74, 0x36, 0x4f, 0x10, 0x1d,
        0x44, 0xd4, 0x73
};

/*  DOCSIS: AES-128 CBC */
static const uint8_t DK3[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t DIV3[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t DP3[] = {
        0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
        0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
        0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
        0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
        0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
        0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
        0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
        0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
};
static const uint8_t DC3[] = {
        0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46,
        0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d,
        0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee,
        0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2,
        0x73, 0xbe, 0xd6, 0xb8, 0xe3, 0xc1, 0x74, 0x3b,
        0x71, 0x16, 0xe6, 0x9e, 0x22, 0x22, 0x95, 0x16,
        0x3f, 0xf1, 0xca, 0xa1, 0x68, 0x1f, 0xac, 0x09,
        0x12, 0x0e, 0xca, 0x30, 0x75, 0x86, 0xe1, 0xa7
};

/*  DOCSIS: AES-256 CFB */
static const uint8_t DK4[] = {
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB
};
static const uint8_t DIV4[] = {
        0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A,
        0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A
};
static const uint8_t DP4[] = {
        0x00, 0x01, 0x02, 0x88, 0xEE, 0x59, 0x7E
};
static const uint8_t DC4[] = {
        0xE3, 0x75, 0xF2, 0x30, 0x1F, 0x75, 0x9A
};

/*  DOCSIS: AES-256 CBC + CFB */
static const uint8_t DK5[] = {
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB,
        0xE6, 0x60, 0x0F, 0xD8, 0x85, 0x2E, 0xF5, 0xAB
};
static const uint8_t DIV5[] = {
        0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A,
        0x81, 0x0E, 0x52, 0x8E, 0x1C, 0x5F, 0xDA, 0x1A
};
static const uint8_t DP5[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x91,
        0xD2, 0xD1, 0x9F
};
static const uint8_t DC5[] = {
        0xD1, 0x28, 0x73, 0x1F, 0xB5, 0x28, 0xB5, 0x18,
        0xAB, 0x51, 0xAB, 0xC8, 0x98, 0x3D, 0xD1, 0xEE,
        0xE4, 0x43, 0x59
};

/*  DOCSIS: AES-256 CBC */
static const uint8_t DK6[] = {
        0xE4, 0x23, 0x33, 0x8A, 0x35, 0x64, 0x61, 0xE2,
        0x49, 0x03, 0xDD, 0xC6, 0xB8, 0xCA, 0x55, 0x7A,
        0xD4, 0xC3, 0xA3, 0xAA, 0x33, 0x62, 0x61, 0xE0,
        0x37, 0x07, 0xB8, 0x23, 0xA2, 0xA3, 0xB5, 0x8D
};
static const uint8_t DIV6[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
};
static const uint8_t DP6[] = {
        0x57, 0x68, 0x61, 0x74, 0x20, 0x61, 0x20, 0x6c,
        0x6f, 0x75, 0x73, 0x79, 0x20, 0x65, 0x61, 0x72,
        0x74, 0x68, 0x21, 0x20, 0x48, 0x65, 0x20, 0x77,
        0x6f, 0x6e, 0x64, 0x65, 0x72, 0x65, 0x64, 0x20,
        0x68, 0x6f, 0x77, 0x20, 0x6d, 0x61, 0x6e, 0x79,
        0x20, 0x70, 0x65, 0x6f, 0x70, 0x6c, 0x65, 0x20,
        0x77, 0x65, 0x72, 0x65, 0x20, 0x64, 0x65, 0x73,
        0x74, 0x69, 0x74, 0x75, 0x74, 0x65, 0x20, 0x74,
        0x68, 0x61, 0x74, 0x20, 0x73, 0x61, 0x6d, 0x65,
        0x20, 0x6e, 0x69, 0x67, 0x68, 0x74, 0x20, 0x65,
        0x76, 0x65, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x68,
        0x69, 0x73, 0x20, 0x6f, 0x77, 0x6e, 0x20, 0x70,
        0x72, 0x6f, 0x73, 0x70, 0x65, 0x72, 0x6f, 0x75,
        0x73, 0x20, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x72,
        0x79, 0x2c, 0x20, 0x68, 0x6f, 0x77, 0x20, 0x6d,
        0x61, 0x6e, 0x79, 0x20, 0x68, 0x6f, 0x6d, 0x65,
        0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x73,
        0x68, 0x61, 0x6e, 0x74, 0x69, 0x65, 0x73, 0x2c,
        0x20, 0x68, 0x6f, 0x77, 0x20, 0x6d, 0x61, 0x6e,
        0x79, 0x20, 0x68, 0x75, 0x73, 0x62, 0x61, 0x6e,
        0x64, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20,
        0x64, 0x72, 0x75, 0x6e, 0x6b, 0x20, 0x61, 0x6e,
        0x64, 0x20, 0x77, 0x69, 0x76, 0x65, 0x73, 0x20,
        0x73, 0x6f, 0x63, 0x6b, 0x65, 0x64, 0x2c, 0x20,
        0x61, 0x6e, 0x64, 0x20, 0x68, 0x6f, 0x77, 0x20,
        0x6d, 0x61, 0x6e, 0x79, 0x20, 0x63, 0x68, 0x69,
        0x6c, 0x64, 0x72, 0x65, 0x6e, 0x20, 0x77, 0x65,
        0x72, 0x65, 0x20, 0x62, 0x75, 0x6c, 0x6c, 0x69,
        0x65, 0x64, 0x2c, 0x20, 0x61, 0x62, 0x75, 0x73,
        0x65, 0x64, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x61,
        0x62, 0x61, 0x6e, 0x64, 0x6f, 0x6e, 0x65, 0x64,
        0x2e, 0x20, 0x48, 0x6f, 0x77, 0x20, 0x6d, 0x61,
        0x6e, 0x79, 0x20, 0x66, 0x61, 0x6d, 0x69, 0x6c,
        0x69, 0x65, 0x73, 0x20, 0x68, 0x75, 0x6e, 0x67,
        0x65, 0x72, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72,
        0x20, 0x66, 0x6f, 0x6f, 0x64, 0x20, 0x74, 0x68,
        0x65, 0x79, 0x20, 0x63, 0x6f, 0x75, 0x6c, 0x64,
        0x20, 0x6e, 0x6f, 0x74, 0x20, 0x61, 0x66, 0x66,
        0x6f, 0x72, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x62,
        0x75, 0x79, 0x3f, 0x20, 0x48, 0x6f, 0x77, 0x20,
        0x6d, 0x61, 0x6e, 0x79, 0x20, 0x68, 0x65, 0x61,
        0x72, 0x74, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65,
        0x20, 0x62, 0x72, 0x6f, 0x6b, 0x65, 0x6e, 0x3f,
        0x20, 0x48, 0x6f, 0x77, 0x20, 0x6d, 0x61, 0x6e,
        0x79, 0x20, 0x73, 0x75, 0x69, 0x63, 0x69, 0x64,
        0x65, 0x73, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64,
        0x20, 0x74, 0x61, 0x6b, 0x65, 0x20, 0x70, 0x6c,
        0x61, 0x63, 0x65, 0x20, 0x74, 0x68, 0x61, 0x74,
        0x20, 0x73, 0x61, 0x6d, 0x65, 0x20, 0x6e, 0x69,
        0x67, 0x68, 0x74, 0x2c, 0x20, 0x68, 0x6f, 0x77,
        0x20, 0x6d, 0x61, 0x6e, 0x79, 0x20, 0x70, 0x65,
        0x6f, 0x70, 0x6c, 0x65, 0x20, 0x77, 0x6f, 0x75,
        0x6c, 0x64, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x6e,
        0x73, 0x61, 0x6e, 0x65, 0x3f, 0x20, 0x48, 0x6f,
        0x77, 0x20, 0x6d, 0x61, 0x6e, 0x79, 0x20, 0x63,
        0x6f, 0x63, 0x6b, 0x72, 0x6f, 0x61, 0x63, 0x68,
        0x65, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x6c,
        0x61, 0x6e, 0x64, 0x6c, 0x6f, 0x72, 0x64, 0x73,
        0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x74,
        0x72, 0x69, 0x75, 0x6d, 0x70, 0x68, 0x3f, 0x20,
        0x48, 0x6f, 0x77, 0x20, 0x6d, 0x61, 0x6e, 0x79,
        0x20, 0x77, 0x69, 0x6e, 0x6e, 0x65, 0x72, 0x73,
        0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x6c, 0x6f,
        0x73, 0x65, 0x72, 0x73, 0x2c, 0x20, 0x73, 0x75
};
static const uint8_t DC6[] = {
        0xF3, 0xDD, 0xF0, 0x0B, 0xFF, 0xA2, 0x6A, 0x04,
        0xBE, 0xDA, 0x52, 0xA6, 0xFE, 0x6B, 0xA6, 0xA7,
        0x48, 0x1D, 0x7D, 0x98, 0x65, 0xDB, 0xEF, 0x06,
        0x26, 0xB5, 0x8E, 0xEB, 0x05, 0x0E, 0x77, 0x98,
        0x17, 0x8E, 0xD0, 0xD4, 0x7B, 0x92, 0x8F, 0x5C,
        0xD0, 0x74, 0x5C, 0xA8, 0x4B, 0x54, 0xB6, 0x2F,
        0x83, 0x72, 0x2C, 0xFF, 0x72, 0xE9, 0xE4, 0x15,
        0x4C, 0x32, 0xAF, 0xC8, 0xC9, 0x89, 0x3C, 0x6E,
        0x31, 0xD5, 0xC0, 0x16, 0xC0, 0x31, 0x7D, 0x11,
        0xAB, 0xCB, 0xDE, 0xD2, 0xD6, 0xAA, 0x76, 0x5E,
        0xBA, 0xF6, 0xE2, 0x92, 0xCB, 0x86, 0x07, 0xFA,
        0xD4, 0x9E, 0x83, 0xED, 0xFD, 0xB8, 0x70, 0x54,
        0x6B, 0xBE, 0xEC, 0x72, 0xDD, 0x28, 0x5E, 0x95,
        0x78, 0xA5, 0x28, 0x43, 0x3D, 0x6D, 0xB1, 0xD9,
        0x69, 0x1F, 0xC9, 0x66, 0x0E, 0x32, 0x44, 0x08,
        0xD2, 0xAE, 0x2C, 0x43, 0xF2, 0xD0, 0x7D, 0x26,
        0x70, 0xE5, 0xA1, 0xCA, 0x37, 0xE9, 0x7D, 0xC7,
        0xA3, 0xFA, 0x81, 0x91, 0x64, 0xAA, 0x64, 0x91,
        0x9A, 0x95, 0x2D, 0xC9, 0xF9, 0xCE, 0xFE, 0x9F,
        0xC4, 0xD8, 0x81, 0xBE, 0x57, 0x84, 0xC5, 0x02,
        0xDB, 0x30, 0xC1, 0xD9, 0x0E, 0xA0, 0xA6, 0x00,
        0xD6, 0xF3, 0x52, 0x7E, 0x0D, 0x23, 0x6B, 0x2B,
        0x34, 0x99, 0x1F, 0x70, 0x27, 0x6D, 0x58, 0x84,
        0x93, 0x77, 0xB8, 0x3E, 0xF1, 0x71, 0x58, 0x42,
        0x8B, 0x2B, 0xC8, 0x6D, 0x05, 0x84, 0xFF, 0x4E,
        0x85, 0xEF, 0x4A, 0x9D, 0x91, 0x6A, 0xD5, 0xE1,
        0xAF, 0x01, 0xEB, 0x83, 0x8F, 0x23, 0x7C, 0x7F,
        0x12, 0x91, 0x05, 0xF0, 0x4E, 0xD9, 0x17, 0x62,
        0x75, 0xBB, 0xAC, 0x97, 0xEE, 0x3B, 0x4E, 0xC7,
        0xE5, 0x92, 0xF8, 0x9D, 0x4C, 0xF9, 0xEE, 0x55,
        0x18, 0xBB, 0xCC, 0xB4, 0xF2, 0x59, 0xB9, 0xFC,
        0x7A, 0x0F, 0x98, 0xD4, 0x8B, 0xFE, 0xF7, 0x83,
        0x46, 0xE2, 0x83, 0x33, 0x3E, 0x95, 0x8D, 0x17,
        0x1E, 0x85, 0xF8, 0x8C, 0x51, 0xB0, 0x6C, 0xB5,
        0x5E, 0x95, 0xBA, 0x4B, 0x69, 0x1B, 0x48, 0x69,
        0x0B, 0x8F, 0xA5, 0x18, 0x13, 0xB9, 0x77, 0xD1,
        0x80, 0x32, 0x32, 0x6D, 0x53, 0xA1, 0x95, 0x40,
        0x96, 0x8A, 0xCC, 0xA3, 0x69, 0xF8, 0x9F, 0xB5,
        0x8E, 0xD2, 0x68, 0x07, 0x4F, 0xA7, 0xEC, 0xF8,
        0x20, 0x21, 0x58, 0xF8, 0xD8, 0x9E, 0x5F, 0x40,
        0xBA, 0xB9, 0x76, 0x57, 0x3B, 0x17, 0xAD, 0xEE,
        0xCB, 0xDF, 0x07, 0xC1, 0xDF, 0x66, 0xA8, 0x0D,
        0xC2, 0xCE, 0x8F, 0x79, 0xC3, 0x32, 0xE0, 0x8C,
        0xFE, 0x5A, 0xF3, 0x55, 0x27, 0x73, 0x6F, 0xA1,
        0x54, 0xC6, 0xFC, 0x28, 0x9D, 0xBE, 0x97, 0xB9,
        0x54, 0x97, 0x72, 0x3A, 0x61, 0xAF, 0x6F, 0xDE,
        0xF8, 0x0E, 0xBB, 0x6B, 0x96, 0x84, 0xDD, 0x9B,
        0x62, 0xBA, 0x47, 0xB5, 0xC9, 0x3B, 0x4E, 0x8C,
        0x78, 0x2A, 0xCC, 0x0A, 0x69, 0x54, 0x25, 0x5E,
        0x8B, 0xAC, 0x56, 0xD9, 0xFE, 0x48, 0xBA, 0xCE,
        0xA9, 0xCE, 0xA6, 0x1D, 0xBF, 0x3E, 0x3C, 0x66,
        0x40, 0x71, 0x79, 0xAD, 0x5B, 0x26, 0xAD, 0xBE,
        0x58, 0x13, 0x64, 0x60, 0x7C, 0x05, 0xFC, 0xE3,
        0x51, 0x7A, 0xF2, 0xCC, 0x54, 0x16, 0x2C, 0xA4,
        0xCE, 0x5F, 0x59, 0x12, 0x77, 0xEB, 0xD9, 0x23,
        0xE3, 0x86, 0xFB, 0xD7, 0x48, 0x76, 0x9D, 0xE3,
        0x89, 0x87, 0x39, 0xFA, 0x7B, 0x21, 0x0B, 0x76,
        0xB2, 0xED, 0x1C, 0x27, 0x4B, 0xD5, 0x27, 0x05,
        0x8C, 0x7D, 0x58, 0x6C, 0xCA, 0xA5, 0x54, 0x9A,
        0x0F, 0xCB, 0xE9, 0x88, 0x31, 0xAD, 0x49, 0xEE,
        0x38, 0xFB, 0xC9, 0xFB, 0xB4, 0x7A, 0x00, 0x58,
        0x20, 0x32, 0xD3, 0x53, 0x5A, 0xDD, 0x74, 0x95,
        0x60, 0x59, 0x09, 0xAE, 0x7E, 0xEC, 0x74, 0xA3,
        0xB7, 0x1C, 0x6D, 0xF2, 0xAE, 0x79, 0xA4, 0x7C
};

static const struct aes_vector docsis_vectors[] = {
        {DK1, DIV1, DP1, sizeof(DP1), DC1, sizeof(DK1)},
        {DK2, DIV2, DP2, sizeof(DP2), DC2, sizeof(DK2)},
        {DK3, DIV3, DP3, sizeof(DP3), DC3, sizeof(DK3)},
        {DK4, DIV4, DP4, sizeof(DP4), DC4, sizeof(DK4)},
        {DK5, DIV5, DP5, sizeof(DP5), DC5, sizeof(DK5)},
        {DK6, DIV6, DP6, sizeof(DP6), DC6, sizeof(DK6)},
};

/* =================================================================== */
/* =================================================================== */
/* Test vectors from CM-SP-SECv3.1-I06-160602 section I.10.2 */

static const uint8_t CFBK1[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef
};
static const uint8_t CFBIV1[] = {
        0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
        0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef
};
static const uint8_t CFBP1[] = {
        0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
        0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20
};
static const uint8_t CFBC1[] = {
        0x43, 0xbc, 0x0a, 0xd0, 0xfc, 0x8d, 0x93, 0xff,
        0x80, 0xe0, 0xbf, 0xf1, 0x41, 0xfc, 0x67, 0x08
};

static const uint8_t CFBK2[] = {
        0xe6, 0x60, 0x0f, 0xd8, 0x85, 0x2e, 0xf5, 0xab,
        0xe6, 0x60, 0x0f, 0xd8, 0x85, 0x2e, 0xf5, 0xab
};
static const uint8_t CFBIV2[] = {
        0x9d, 0xd1, 0x67, 0x4b, 0xba, 0x61, 0x10, 0x1b,
        0x56, 0x75, 0x64, 0x74, 0x36, 0x4f, 0x10, 0x1d
};
static const uint8_t CFBP2[] = {
        0xd2, 0xd1, 0x9f
};
static const uint8_t CFBC2[] = {
        0x44, 0xd4, 0x73
};

/* =================================================================== */
/* =================================================================== */

/*
 * Test vectors from
 * https://nvlpubs.nist.gov/nistpubs/Legacy/SP/nistspecialpublication800-38a.pdf
 */
static const uint8_t CFBK3[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t CFBIV3[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
};
static const uint8_t CFBP3[] = {
        0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
        0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a
};
static const uint8_t CFBC3[] = {
        0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20,
        0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a
};

static const uint8_t CFBK4[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t CFBIV4[] = {
        0x3b, 0x3f, 0xd9, 0x2e, 0xb7, 0x2d, 0xad, 0x20,
        0x33, 0x34, 0x49, 0xf8, 0xe8, 0x3c, 0xfb, 0x4a
};
static const uint8_t CFBP4[] = {
        0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
        0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51
};
static const uint8_t CFBC4[] = {
        0xc8, 0xa6, 0x45, 0x37, 0xa0, 0xb3, 0xa9, 0x3f,
        0xcd, 0xe3, 0xcd, 0xad, 0x9f, 0x1c, 0xe5, 0x8b
};

static const uint8_t CFBK5[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t CFBIV5[] = {
        0xc8, 0xa6, 0x45, 0x37, 0xa0, 0xb3, 0xa9, 0x3f,
        0xcd, 0xe3, 0xcd, 0xad, 0x9f, 0x1c, 0xe5, 0x8b
};
static const uint8_t CFBP5[] = {
        0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
        0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef
};
static const uint8_t CFBC5[] = {
        0x26, 0x75, 0x1f, 0x67, 0xa3, 0xcb, 0xb1, 0x40,
        0xb1, 0x80, 0x8c, 0xf1, 0x87, 0xa4, 0xf4, 0xdf
};

static const uint8_t CFBK6[] = {
        0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
        0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
};
static const uint8_t CFBIV6[] = {
        0x26, 0x75, 0x1f, 0x67, 0xa3, 0xcb, 0xb1, 0x40,
        0xb1, 0x80, 0x8c, 0xf1, 0x87, 0xa4, 0xf4, 0xdf
};
static const uint8_t CFBP6[] = {
        0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
        0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10
};
static const uint8_t CFBC6[] = {
        0xc0, 0x4b, 0x05, 0x35, 0x7c, 0x5d, 0x1c, 0x0e,
        0xea, 0xc4, 0xc6, 0x6f, 0x9f, 0xf7, 0xf2, 0xe6
};
static struct aes_vector aes_cfb_128_tab[] = {
        {CFBK1, CFBIV1, CFBP1, sizeof(CFBP1), CFBC1, sizeof(CFBK1)},
        {CFBK2, CFBIV2, CFBP2, sizeof(CFBP2), CFBC2, sizeof(CFBK2)},
        {CFBK3, CFBIV3, CFBP3, sizeof(CFBP3), CFBC3, sizeof(CFBK3)},
        {CFBK4, CFBIV4, CFBP4, sizeof(CFBP4), CFBC4, sizeof(CFBK4)},
        {CFBK5, CFBIV5, CFBP5, sizeof(CFBP5), CFBC5, sizeof(CFBK5)},
        {CFBK6, CFBIV6, CFBP6, sizeof(CFBP6), CFBC6, sizeof(CFBK6)},
};

/* =================================================================== */
/* =================================================================== */
/* DOCSIS with CRC: AES CBC + CFB + CRC32 */

/* 128-bit keys */
static const uint8_t DOCRC1_KEY[] = {
        0x00, 0x00, 0x00, 0x00, 0xaa, 0xbb, 0xcc, 0xdd,
        0xee, 0xff, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55
};
static const uint8_t DOCRC1_IV[] = {
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
};
static const uint8_t DOCRC1_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (14 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00,
        /* CRC (4 bytes) */
        0x14, 0x08, 0xe8, 0x55
};
static const uint8_t DOCRC1_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x06, 0x05,
        0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x7A, 0xF0,
        /* CRC (encrypted) */
        0x61, 0xF8, 0x63, 0x42
};
#define DOCRC1_KEY_LEN       16
#define DOCRC1_HASH_OFFSET   6
#define DOCRC1_HASH_LENGTH   14
#define DOCRC1_CIPHER_OFFSET 18
#define DOCRC1_CIPHER_LENGTH 6
#define DOCRC1_CRC           0x55e80814 /* LE */
#define DOCRC1_FRAME_LEN     DIM(DOCRC1_PT)

static const uint8_t DOCRC2_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA,
        /* CRC (4 bytes) */
        0x0E, 0x99, 0x8E, 0xFE
};
static const uint8_t DOCRC2_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x7A, 0xF0,
        /* PDU payload */
        0xDF,
        /* CRC (encrypted) */
        0xFE, 0x12, 0x99, 0xE5
};
#define DOCRC2_KEY           DOCRC1_KEY
#define DOCRC2_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC2_IV            DOCRC1_IV
#define DOCRC2_HASH_OFFSET   6
#define DOCRC2_HASH_LENGTH   15
#define DOCRC2_CIPHER_OFFSET 18
#define DOCRC2_CIPHER_LENGTH 7
#define DOCRC2_CRC           0xFE8E990E
#define DOCRC2_FRAME_LEN     DIM(DOCRC2_PT)

static const uint8_t DOCRC3_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xCB, 0x7C, 0xAB, 0x56
};
static const uint8_t DOCRC3_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0xD6, 0xE2,
        /* PDU payload */
        0x70, 0x5C, 0xE6, 0x4D, 0xCC, 0x8C, 0x47, 0xB7,
        0x09, 0xD6,
        /* CRC (encrypted) */
        0x54, 0x85, 0xF8, 0x32
};
#define DOCRC3_KEY           DOCRC1_KEY
#define DOCRC3_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC3_IV            DOCRC1_IV
#define DOCRC3_HASH_OFFSET   6
#define DOCRC3_HASH_LENGTH   24
#define DOCRC3_CIPHER_OFFSET 18
#define DOCRC3_CIPHER_LENGTH 16
#define DOCRC3_CRC           0x56AB7CCB
#define DOCRC3_FRAME_LEN     DIM(DOCRC3_PT)

static const uint8_t DOCRC4_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA,
        /* CRC (4 bytes) */
        0x3F, 0x15, 0xE1, 0xE8
};
static const uint8_t DOCRC4_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x92, 0x6A,
        /* PDU payload */
        0xC2, 0xDC, 0xEE, 0x3B, 0x31, 0xEC, 0x03, 0xDE,
        0x95, 0x33, 0x5E,
        /* CRC (encrypted) */
        0xFE, 0x47, 0x3E, 0x22
};
#define DOCRC4_KEY           DOCRC1_KEY
#define DOCRC4_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC4_IV            DOCRC1_IV
#define DOCRC4_HASH_OFFSET   6
#define DOCRC4_HASH_LENGTH   25
#define DOCRC4_CIPHER_OFFSET 18
#define DOCRC4_CIPHER_LENGTH 17
#define DOCRC4_CRC           0xE8E1153F
#define DOCRC4_FRAME_LEN     DIM(DOCRC4_PT)

static const uint8_t DOCRC5_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
        /* CRC (4 bytes) */
        0x2E, 0x07, 0xC8, 0x3C
};
static const uint8_t DOCRC5_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x77, 0x74,
        /* PDU payload */
        0x56, 0x05, 0xD1, 0x14, 0xA2, 0x8D, 0x2C, 0x9A,
        0x11, 0xFC, 0x7D, 0xB0, 0xE7, 0x18, 0xCE, 0x75,
        0x7C, 0x89, 0x14, 0x56, 0xE2, 0xF2, 0xB7, 0x47,
        0x08, 0x27, 0xF7, 0x08, 0x7A, 0x13, 0x90, 0x81,
        0x75, 0xB0, 0xC7, 0x91, 0x04, 0x83, 0xAD, 0x11,
        0x46, 0x46, 0xF8, 0x54, 0x87, 0xA0, 0x42, 0xF3,
        0x71, 0xA9, 0x8A, 0xCD, 0x59, 0x77, 0x67, 0x11,
        0x1A, 0x87,
        /* CRC (encrypted) */
        0xAB, 0xED, 0x2C, 0x26
};
#define DOCRC5_KEY           DOCRC1_KEY
#define DOCRC5_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC5_IV            DOCRC1_IV
#define DOCRC5_HASH_OFFSET   6
#define DOCRC5_HASH_LENGTH   72
#define DOCRC5_CIPHER_OFFSET 18
#define DOCRC5_CIPHER_LENGTH 64
#define DOCRC5_CRC           0x3CC8072E
#define DOCRC5_FRAME_LEN     DIM(DOCRC5_PT)

static const uint8_t DOCRC6_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xB3, 0x60, 0xEB, 0x38
};
static const uint8_t DOCRC6_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x77, 0x74,
        /* PDU payload */
        0x56, 0x05, 0xD1, 0x14, 0xA2, 0x8D, 0x2C, 0x9A,
        0x11, 0xFC, 0x7D, 0xB0, 0xE7, 0x18, 0xCE, 0x75,
        0x7C, 0x89, 0x14, 0x56, 0xE2, 0xF2, 0xB7, 0x47,
        0x08, 0x27, 0xF7, 0x08, 0x7A, 0x13, 0x90, 0x81,
        0x75, 0xB0, 0xC7, 0x91, 0x04, 0x83, 0xAD, 0x11,
        0x46, 0x46, 0xF8, 0x54, 0x87, 0xA0, 0xA4, 0x0C,
        0xC2, 0xF0, 0x81, 0x49, 0xA8, 0xA6, 0x6C, 0x48,
        0xEB, 0x1F, 0x4B,
        /* CRC (encrypted) */
        0x2F, 0xD4, 0x48, 0x18
};
#define DOCRC6_KEY           DOCRC1_KEY
#define DOCRC6_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC6_IV            DOCRC1_IV
#define DOCRC6_HASH_OFFSET   6
#define DOCRC6_HASH_LENGTH   73
#define DOCRC6_CIPHER_OFFSET 18
#define DOCRC6_CIPHER_LENGTH 65
#define DOCRC6_CRC           0x38EB60B3
#define DOCRC6_FRAME_LEN     DIM(DOCRC6_PT)

static const uint8_t DOCRC7_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xB3, 0x60, 0xEB, 0x38
};
static const uint8_t DOCRC7_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0x3B, 0x9F, 0x72, 0x4C,
        0xB5, 0x72, 0x3E, 0x56, 0x54, 0x49, 0x13, 0x53,
        0xC4, 0xAA, 0xCD, 0xEA, 0x6A, 0x88, 0x99, 0x07,
        0x86, 0xF4, 0xCF, 0x03, 0x4E, 0xDF, 0x65, 0x61,
        0x47, 0x5B, 0x2F, 0x81, 0x09, 0x12, 0x9A, 0xC2,
        0x24, 0x8C, 0x09,
        /* CRC (encrypted) */
        0x11, 0xB4, 0x06, 0x33
};
#define DOCRC7_KEY           DOCRC1_KEY
#define DOCRC7_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC7_IV            DOCRC1_IV
#define DOCRC7_HASH_OFFSET   6
#define DOCRC7_HASH_LENGTH   73
#define DOCRC7_CIPHER_OFFSET 40
#define DOCRC7_CIPHER_LENGTH 43
#define DOCRC7_CRC           0x38EB60B3
#define DOCRC7_FRAME_LEN     DIM(DOCRC7_PT)

/* no cipher / CRC only */
static const uint8_t DOCRC8_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xCB, 0x7C, 0xAB, 0x56
};
static const uint8_t DOCRC8_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xCB, 0x7C, 0xAB, 0x56
};
#define DOCRC8_KEY           DOCRC1_KEY
#define DOCRC8_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC8_IV            DOCRC1_IV
#define DOCRC8_HASH_OFFSET   6
#define DOCRC8_HASH_LENGTH   24
#define DOCRC8_CIPHER_OFFSET 18
#define DOCRC8_CIPHER_LENGTH 0
#define DOCRC8_CRC           0x56AB7CCB
#define DOCRC8_FRAME_LEN     DIM(DOCRC8_PT)

/* cipher only / no CRC */
static const uint8_t DOCRC9_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
        /* instead of CRC (4 bytes) */
        0xAA, 0xAA, 0xAA, 0xAA
};
static const uint8_t DOCRC9_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* DA/ SA */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x77, 0x74,
        /* PDU payload */
        0x56, 0x05, 0xD1, 0x14, 0xA2, 0x8D, 0x2C, 0x9A,
        0x11, 0xFC,
        /* instead of CRC (4 bytes) */
        0x7D, 0xB0, 0xE7, 0x18
};
#define DOCRC9_KEY           DOCRC1_KEY
#define DOCRC9_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC9_IV            DOCRC1_IV
#define DOCRC9_HASH_OFFSET   6
#define DOCRC9_HASH_LENGTH   0
#define DOCRC9_CIPHER_OFFSET 18
#define DOCRC9_CIPHER_LENGTH 16
#define DOCRC9_CRC           0x0
#define DOCRC9_FRAME_LEN     DIM(DOCRC9_PT)

/* no cipher / no CRC */
static const uint8_t DOCRC10_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xFF, 0xFF, 0xFF, 0x55
};
static const uint8_t DOCRC10_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len */
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xFF, 0xFF, 0xFF, 0x55
};
#define DOCRC10_KEY           DOCRC1_KEY
#define DOCRC10_KEY_LEN       DOCRC1_KEY_LEN
#define DOCRC10_IV            DOCRC1_IV
#define DOCRC10_HASH_OFFSET   6
#define DOCRC10_HASH_LENGTH   0
#define DOCRC10_CIPHER_OFFSET 18
#define DOCRC10_CIPHER_LENGTH 0
#define DOCRC10_CRC           0x0
#define DOCRC10_FRAME_LEN     DIM(DOCRC10_PT)

static const uint8_t DOCRC18_KEY[] = {
        0x00, 0x00, 0x00, 0x00, 0xaa, 0xbb, 0xcc, 0xdd,
        0xee, 0xff, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55
};
static const uint8_t DOCRC18_IV[] = {
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
};
static const uint8_t DOCRC18_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (14 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00,
        /* PDU payload */
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xFF,
        0xFF, 0xFF, 0xFF
};
static const uint8_t DOCRC18_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU payload */
        0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA,
        0x3B, 0x9F, 0x72, 0x4C, 0xB5, 0x72,
        0x3E, 0x56, 0x54, 0x49, 0x13, 0x53,
        0xC4, 0xAA, 0xCD, 0xEA, 0x6A, 0x88,
        0x99, 0x07, 0x86, 0xF4, 0xCF, 0x03,
        0x4E, 0xDF, 0x65, 0x61, 0x47, 0x5B,
        0x2F, 0x81, 0x09, 0x12, 0x9A, 0xC2,
        0x24, 0x8C, 0x09, 0x5D, 0x2B, 0x12,
        0xF4
};
#define DOCRC18_KEY_LEN       16
#define DOCRC18_HASH_OFFSET   6
#define DOCRC18_HASH_LENGTH   0
#define DOCRC18_CIPHER_OFFSET 40
#define DOCRC18_CIPHER_LENGTH 43
#define DOCRC18_CRC           0x0
#define DOCRC18_FRAME_LEN     DIM(DOCRC18_PT)

/* 256-bit keys */
static const uint8_t DOCRC11_KEY[] = {
        0x00, 0x00, 0x00, 0x00, 0xAA, 0xBB, 0xCC, 0xDD,
        0xEE, 0xFF, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55,
        0x00, 0x00, 0x00, 0x00, 0xAA, 0xBB, 0xCC, 0xDD,
        0xEE, 0xFF, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55
};
static const uint8_t DOCRC11_IV[] = {
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
};
static const uint8_t DOCRC11_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (14 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00,
        /* CRC (4 bytes) */
        0x14, 0x08, 0xe8, 0x55
};
static const uint8_t DOCRC11_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x08, 0x46,
        /* CRC (encrypted) */
        0x98, 0x98, 0x47, 0x76
};

#define DOCRC11_KEY_LEN       32
#define DOCRC11_HASH_OFFSET   6
#define DOCRC11_HASH_LENGTH   14
#define DOCRC11_CIPHER_OFFSET 18
#define DOCRC11_CIPHER_LENGTH 6
#define DOCRC11_CRC           0x55e80814 /* LE */
#define DOCRC11_FRAME_LEN     DIM(DOCRC11_PT)

static const uint8_t DOCRC12_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (15 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00, 0xAA,
        /* CRC (4 bytes) */
        0x0E, 0x99, 0x8E, 0xFE
};
static const uint8_t DOCRC12_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x08, 0x46, 0x26,
        /* CRC (encrypted) */
        0x9E, 0x36, 0xAD, 0xC7
};

#define DOCRC12_KEY           DOCRC11_KEY
#define DOCRC12_KEY_LEN       DOCRC11_KEY_LEN
#define DOCRC12_IV            DOCRC11_IV
#define DOCRC12_HASH_OFFSET   6
#define DOCRC12_HASH_LENGTH   15
#define DOCRC12_CIPHER_OFFSET 18
#define DOCRC12_CIPHER_LENGTH 7
#define DOCRC12_CRC           0xFE8E990E /* LE */
#define DOCRC12_FRAME_LEN     DIM(DOCRC12_PT)

static const uint8_t DOCRC13_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (24 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        /* CRC (4 bytes) */
        0xCB, 0x7C, 0xAB, 0x56
};
static const uint8_t DOCRC13_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0xA1, 0xFF, 0x45, 0x21, 0x43, 0xC7,
        0x5B, 0x3C, 0xAB, 0x7A, 0x80, 0xC9,
        /* CRC (encrypted) */
        0x70, 0x51, 0x54, 0xD0
};

#define DOCRC13_KEY           DOCRC11_KEY
#define DOCRC13_KEY_LEN       DOCRC11_KEY_LEN
#define DOCRC13_IV            DOCRC11_IV
#define DOCRC13_HASH_OFFSET   6
#define DOCRC13_HASH_LENGTH   24
#define DOCRC13_CIPHER_OFFSET 18
#define DOCRC13_CIPHER_LENGTH 16
#define DOCRC13_CRC           0x56AB7CCB /* LE */
#define DOCRC13_FRAME_LEN     DIM(DOCRC13_PT)

static const uint8_t DOCRC14_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (24 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA,
        /* CRC (4 bytes) */
        0x3F, 0x15, 0xE1, 0xE8
};
static const uint8_t DOCRC14_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0xA0, 0x88, 0x19, 0x50, 0x3B, 0x70,
        0x91, 0x3C, 0x2A, 0x46, 0x74, 0x45,
        0x03,
        /* CRC (encrypted) */
        0x30, 0xC8, 0xA9, 0xCC
};

#define DOCRC14_KEY           DOCRC11_KEY
#define DOCRC14_KEY_LEN       DOCRC11_KEY_LEN
#define DOCRC14_IV            DOCRC11_IV
#define DOCRC14_HASH_OFFSET   6
#define DOCRC14_HASH_LENGTH   25
#define DOCRC14_CIPHER_OFFSET 18
#define DOCRC14_CIPHER_LENGTH 17
#define DOCRC14_CRC           0xE8E1153F /* LE */
#define DOCRC14_FRAME_LEN     DIM(DOCRC14_PT)

static const uint8_t DOCRC15_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (72 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        /* CRC (4 bytes) */
        0x2E, 0x07, 0xC8, 0x3C
};

static const uint8_t DOCRC15_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x5D, 0xB3, 0xD1, 0xF4, 0x6C, 0x65,
        0xCA, 0x3B, 0x5C, 0x16, 0xB3, 0xA4,
        0x23, 0x74, 0x81, 0x5E, 0x12, 0x03,
        0x7B, 0x3F, 0xBB, 0x62, 0x1D, 0x29,
        0x66, 0x60, 0x1B, 0x6E, 0x01, 0xFE,
        0x6F, 0x40, 0x12, 0xE6, 0x20, 0xE6,
        0x10, 0xBE, 0x5B, 0xF2, 0x7E, 0x7F,
        0x43, 0x53, 0x66, 0x38, 0xA6, 0x4D,
        0xF3, 0x66, 0x84, 0x9F, 0xE2, 0xEC,
        0x9F, 0xBC, 0xD4, 0x38, 0xDB, 0x33,
        /* CRC (encrypted) */
        0x4E, 0x12, 0xB3, 0xB4
};

#define DOCRC15_KEY           DOCRC11_KEY
#define DOCRC15_KEY_LEN       DOCRC11_KEY_LEN
#define DOCRC15_IV            DOCRC11_IV
#define DOCRC15_HASH_OFFSET   6
#define DOCRC15_HASH_LENGTH   72
#define DOCRC15_CIPHER_OFFSET 18
#define DOCRC15_CIPHER_LENGTH 64
#define DOCRC15_CRC           0x3CC8072E /* LE */
#define DOCRC15_FRAME_LEN     DIM(DOCRC15_PT)

static const uint8_t DOCRC16_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (72 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA,
        /* CRC (4 bytes) */
        0xB3, 0x60, 0xEB, 0x38
};

static const uint8_t DOCRC16_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA (12 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        /* PDU Type/Len (encrypted) */
        0x5D, 0xB3, 0xD1, 0xF4, 0x6C, 0x65,
        0xCA, 0x3B, 0x5C, 0x16, 0xB3, 0xA4,
        0x23, 0x74, 0x81, 0x5E, 0x12, 0x03,
        0x7B, 0x3F, 0xBB, 0x62, 0x1D, 0x29,
        0x66, 0x60, 0x1B, 0x6E, 0x01, 0xFE,
        0x6F, 0x40, 0x12, 0xE6, 0x20, 0xE6,
        0x10, 0xBE, 0x5B, 0xF2, 0x7E, 0x7F,
        0x43, 0x53, 0x66, 0x38, 0xA6, 0x4D,
        0xD9, 0xA6, 0x6A, 0x07, 0x6B, 0xAA,
        0x5C, 0xF6, 0xB4, 0x1D, 0xC5, 0x9A,
        0x7C,
        /* CRC (encrypted) */
        0x48, 0xDB, 0xB1, 0x74
};

#define DOCRC16_KEY           DOCRC11_KEY
#define DOCRC16_KEY_LEN       DOCRC11_KEY_LEN
#define DOCRC16_IV            DOCRC11_IV
#define DOCRC16_HASH_OFFSET   6
#define DOCRC16_HASH_LENGTH   73
#define DOCRC16_CIPHER_OFFSET 18
#define DOCRC16_CIPHER_LENGTH 65
#define DOCRC16_CRC           0x38EB60B3 /* LE */
#define DOCRC16_FRAME_LEN     DIM(DOCRC16_PT)

static const uint8_t DOCRC17_PT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU without CRC (73 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA,
        /* CRC (4 bytes) */
        0xB3, 0x60, 0xEB, 0x38
};

static const uint8_t DOCRC17_CT[] = {
        /* DOCSIS Header (6 bytes) */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        /* PDU DA / SA + Extra data (34 bytes) */
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x06, 0x05, 0x04, 0x03, 0x02, 0x01,
        0x08, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xAA, 0xAA, 0xAA, 0xAA,
        /* PDU Type/Len (encrypted) */
        0x96, 0x03, 0x94, 0x9B, 0xDA, 0x96,
        0x00, 0x42, 0x7B, 0x52, 0xD3, 0xB9,
        0xA4, 0x10, 0x7B, 0x87, 0x0B, 0xBA,
        0x41, 0x0E, 0x2B, 0x8F, 0xA6, 0xA3,
        0xF5, 0x55, 0x9C, 0x0C, 0x69, 0x7C,
        0x36, 0xD2, 0xBF, 0xA1, 0xF2, 0x2B,
        0xAB, 0x1A, 0x92,
        /* CRC (encrypted) */
        0xAF, 0x19, 0x88, 0xDD
};

#define DOCRC17_KEY           DOCRC11_KEY
#define DOCRC17_KEY_LEN       DOCRC11_KEY_LEN
#define DOCRC17_IV            DOCRC11_IV
#define DOCRC17_HASH_OFFSET   6
#define DOCRC17_HASH_LENGTH   73
#define DOCRC17_CIPHER_OFFSET 40
#define DOCRC17_CIPHER_LENGTH 43
#define DOCRC17_CRC           0x38EB60B3 /* LE */
#define DOCRC17_FRAME_LEN     DIM(DOCRC17_PT)


#define MK_DOCRC_VEC(_n)                                                \
        { _n ## _FRAME_LEN, _n ## _KEY, _n ## _KEY_LEN, _n ## _IV,      \
                        _n ## _PT, _n ## _CT,                           \
                        _n ## _HASH_OFFSET, _n ## _HASH_LENGTH,         \
                        _n ## _CIPHER_OFFSET, _n ## _CIPHER_LENGTH,     \
                        _n ## _CRC }

struct docsis_crc_vector {
        uint64_t frame_len;
        const uint8_t *key;
        uint64_t key_len;
        const uint8_t *iv;
        const uint8_t *pt;
        const uint8_t *ct;
        uint64_t hash_offset;
        uint64_t hash_length;
        uint64_t cipher_offset;
        uint64_t cipher_length;
        uint32_t crc_hash;
};

struct docsis_crc_vector docsis_crc_tab[] = {
        /* 128-bit key */
        MK_DOCRC_VEC(DOCRC1),
        MK_DOCRC_VEC(DOCRC2),
        MK_DOCRC_VEC(DOCRC3),
        MK_DOCRC_VEC(DOCRC4),
        MK_DOCRC_VEC(DOCRC5),
        MK_DOCRC_VEC(DOCRC6),
        MK_DOCRC_VEC(DOCRC7),
        MK_DOCRC_VEC(DOCRC8),
        MK_DOCRC_VEC(DOCRC9),
        MK_DOCRC_VEC(DOCRC10),
        MK_DOCRC_VEC(DOCRC18),
        /* 256-bit key */
        MK_DOCRC_VEC(DOCRC11),
        MK_DOCRC_VEC(DOCRC12),
        MK_DOCRC_VEC(DOCRC13),
        MK_DOCRC_VEC(DOCRC14),
        MK_DOCRC_VEC(DOCRC15),
        MK_DOCRC_VEC(DOCRC16),
        MK_DOCRC_VEC(DOCRC17),
};

static int
aes_job_ok(const struct IMB_JOB *job,
           const uint8_t *out_text,
           const uint8_t *target,
           const uint8_t *padding,
           const size_t sizeof_padding,
           const unsigned text_len)
{
        const int num = (const int)((uint64_t)job->user_data2);

        if (job->status != STS_COMPLETED) {
                printf("%d error status:%d, job %d",
                       __LINE__, job->status, num);
                return 0;
        }
        if (memcmp(out_text, target + sizeof_padding,
                   text_len)) {
                printf("%d mismatched\n", num);
                return 0;
        }
        if (memcmp(padding, target, sizeof_padding)) {
                printf("%d overwrite head\n", num);
                return 0;
        }
        if (memcmp(padding,
                   target + sizeof_padding + text_len,
                   sizeof_padding)) {
                printf("%d overwrite tail\n", num);
                return 0;
        }
        return 1;
}

static int
test_aes_many(struct IMB_MGR *mb_mgr,
              void *enc_keys,
              void *dec_keys,
              const void *iv,
              const uint8_t *in_text,
              const uint8_t *out_text,
              const unsigned text_len,
              const int dir,
              const int order,
              const JOB_CIPHER_MODE cipher,
              const int in_place,
              const int key_len,
              const int num_jobs)
{
        struct IMB_JOB *job;
        uint8_t padding[16];
        uint8_t **targets = malloc(num_jobs * sizeof(void *));
        int i, err, jobs_rx = 0, ret = -1;

        if (targets == NULL)
                goto end_alloc;

        memset(targets, 0, num_jobs * sizeof(void *));
        memset(padding, -1, sizeof(padding));

        for (i = 0; i < num_jobs; i++) {
                targets[i] = malloc(text_len + (sizeof(padding) * 2));
                if (targets[i] == NULL)
                        goto end_alloc;
                memset(targets[i], -1, text_len + (sizeof(padding) * 2));
                if (in_place) {
                        /* copy input text to the allocated buffer */
                        memcpy(targets[i] + sizeof(padding), in_text, text_len);
                }
        }

        /* flush the scheduler */
        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL)
                ;

        for (i = 0; i < num_jobs; i++) {
                job = IMB_GET_NEXT_JOB(mb_mgr);
                job->cipher_direction = dir;
                job->chain_order = order;
                if (!in_place) {
                        job->dst = targets[i] + sizeof(padding);
                        job->src = in_text;
                } else {
                        job->dst = targets[i] + sizeof(padding);
                        job->src = targets[i] + sizeof(padding);
                }
                job->cipher_mode = cipher;
                job->enc_keys = enc_keys;
                job->dec_keys = dec_keys;
                job->key_len_in_bytes = key_len;

                job->iv = iv;
                job->iv_len_in_bytes = 16;
                job->cipher_start_src_offset_in_bytes = 0;
                job->msg_len_to_cipher_in_bytes = text_len;
                job->user_data = targets[i];
                job->user_data2 = (void *)((uint64_t)i);

                job->hash_alg = IMB_AUTH_NULL;

                job = IMB_SUBMIT_JOB(mb_mgr);
                if (job == NULL) {
                        /* no job returned - check for error */
                        err = imb_get_errno(mb_mgr);
                        if (err != 0) {
                                printf("Error: %s!\n", imb_get_strerror(err));
                                goto end;
                        }
                } else {
                        /* got job back */
                        jobs_rx++;
                        if (!aes_job_ok(job, out_text, job->user_data, padding,
                                       sizeof(padding), text_len))
                                goto end;
                }
        }

        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL) {
                err = imb_get_errno(mb_mgr);
                if (err != 0) {
                        printf("Error: %s!\n", imb_get_strerror(err));
                        goto end;
                }

                jobs_rx++;
                if (!aes_job_ok(job, out_text, job->user_data, padding,
                               sizeof(padding), text_len))
                        goto end;
        }

        if (jobs_rx != num_jobs) {
                printf("Expected %d jobs, received %d\n", num_jobs, jobs_rx);
                goto end;
        }
        ret = 0;

 end:
        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL) {
                err = imb_get_errno(mb_mgr);
                if (err != 0) {
                        printf("Error: %s!\n", imb_get_strerror(err));
                        goto end;
                }
        }

end_alloc:
        if (targets != NULL) {
                for (i = 0; i < num_jobs; i++)
                        free(targets[i]);
                free(targets);
        }

        return ret;
}

static void
test_aes_vectors(struct IMB_MGR *mb_mgr,
                 struct test_suite_context *ctx128,
                 struct test_suite_context *ctx192,
                 struct test_suite_context *ctx256,
                 const int vec_cnt,
                 const struct aes_vector *vec_tab, const char *banner,
                 const JOB_CIPHER_MODE cipher, const int num_jobs)
{
	int vect;
        DECLARE_ALIGNED(uint32_t enc_keys[15*4], 16);
        DECLARE_ALIGNED(uint32_t dec_keys[15*4], 16);

	printf("%s (N jobs = %d):\n", banner, num_jobs);
	for (vect = 0; vect < vec_cnt; vect++) {
                struct test_suite_context *ctx;
#ifdef DEBUG
		printf("[%d/%d] Standard vector key_len:%d\n",
                       vect + 1, vec_cnt,
                       (int) vec_tab[vect].Klen);
#else
		printf(".");
#endif
                switch (vec_tab[vect].Klen) {
                case 16:
                        IMB_AES_KEYEXP_128(mb_mgr, vec_tab[vect].K, enc_keys,
                                           dec_keys);
                        ctx = ctx128;
                        break;
                case 24:
                        IMB_AES_KEYEXP_192(mb_mgr, vec_tab[vect].K, enc_keys,
                                           dec_keys);
                        ctx = ctx192;
                        break;
                case 32:
                default:
                        IMB_AES_KEYEXP_256(mb_mgr, vec_tab[vect].K, enc_keys,
                                           dec_keys);
                        ctx = ctx256;
                        break;
                }

                if (test_aes_many(mb_mgr, enc_keys, dec_keys,
                                  vec_tab[vect].IV,
                                  vec_tab[vect].P, vec_tab[vect].C,
                                  (unsigned) vec_tab[vect].Plen,
                                  IMB_DIR_ENCRYPT, IMB_ORDER_CIPHER_HASH,
                                  cipher, 0,
                                  vec_tab[vect].Klen, num_jobs)) {
                        printf("error #%d encrypt\n", vect + 1);
                        test_suite_update(ctx, 0, 1);
                } else {
                        test_suite_update(ctx, 1, 0);
                }

                if (test_aes_many(mb_mgr, enc_keys, dec_keys,
                                  vec_tab[vect].IV,
                                  vec_tab[vect].C, vec_tab[vect].P,
                                  (unsigned) vec_tab[vect].Plen,
                                  IMB_DIR_DECRYPT, IMB_ORDER_HASH_CIPHER,
                                  cipher, 0,
                                  vec_tab[vect].Klen, num_jobs)) {
                        printf("error #%d decrypt\n", vect + 1);
                        test_suite_update(ctx, 0, 1);
                } else {
                        test_suite_update(ctx, 1, 0);
                }

                if (test_aes_many(mb_mgr, enc_keys, dec_keys,
                                  vec_tab[vect].IV,
                                  vec_tab[vect].P, vec_tab[vect].C,
                                  (unsigned) vec_tab[vect].Plen,
                                  IMB_DIR_ENCRYPT, IMB_ORDER_CIPHER_HASH,
                                  cipher, 1,
                                  vec_tab[vect].Klen, num_jobs)) {
                        printf("error #%d encrypt in-place\n", vect + 1);
                        test_suite_update(ctx, 0, 1);
                } else {
                        test_suite_update(ctx, 1, 0);
                }

                if (test_aes_many(mb_mgr, enc_keys, dec_keys,
                                  vec_tab[vect].IV,
                                  vec_tab[vect].C, vec_tab[vect].P,
                                  (unsigned) vec_tab[vect].Plen,
                                  IMB_DIR_DECRYPT, IMB_ORDER_HASH_CIPHER,
                                  cipher, 1,
                                  vec_tab[vect].Klen, num_jobs)) {
                        printf("error #%d decrypt in-place\n", vect + 1);
                        test_suite_update(ctx, 0, 1);
                } else {
                        test_suite_update(ctx, 1, 0);
                }
	}
	printf("\n");
}

static int
docrc_job_ok(const struct IMB_JOB *job,
             const struct docsis_crc_vector *p_vec,
             const uint8_t *target,
             const uint8_t *padding,
             const size_t sizeof_padding,
             const uint8_t *out_text)
{
        const int num = (const int)((uint64_t)job->user_data2);
        const unsigned text_len = (unsigned) p_vec->frame_len;

        if (job->status != STS_COMPLETED) {
                printf("%d error status:%d, job %d",
                       __LINE__, job->status, num);
                return 0;
        }

        if (job->msg_len_to_cipher_in_bytes > 0) {
                if (memcmp(out_text, target + sizeof_padding,
                           text_len)) {
                        printf("%d message mismatched\n", num);
                        return 0;
                }

                if (memcmp(padding, target, sizeof_padding)) {
                        printf("%d overwrite head\n", num);
                        return 0;
                }

                if (memcmp(padding,
                           target + sizeof_padding + text_len,
                           sizeof_padding)) {
                        printf("%d overwrite tail\n", num);
                        return 0;
                }
        }

        if (job->msg_len_to_hash_in_bytes >= 14) {
                if (memcmp(job->auth_tag_output, &p_vec->crc_hash,
                           sizeof(p_vec->crc_hash))) {
                        const uint32_t *p_got =
                                (const uint32_t *)job->auth_tag_output;
                        const uint32_t *p_exp =
                                (const uint32_t *)&p_vec->crc_hash;

                        printf("%d authentication tag mismatch "
                               "(got = 0x%X, exp = 0x%X)\n",
                               num, *p_got, *p_exp);
                        return 0;
                }
        }

        return 1;
}

static int
test_docrc_many(struct IMB_MGR *mb_mgr,
                void *enc_keys,
                void *dec_keys,
                const struct docsis_crc_vector *p_vec,
                const int dir,
                const int order,
                const int in_place,
                const int num_jobs)
{
        const uint64_t key_len = p_vec->key_len;
        const unsigned tag_len = 4;
        const unsigned frame_len = (unsigned) p_vec->frame_len;
        struct IMB_JOB *job;
        uint8_t padding[16];
        uint8_t **targets = malloc(num_jobs * sizeof(void *));
        uint32_t *auths = malloc(num_jobs * sizeof(uint32_t));
        int i, jobs_rx = 0, ret = -1;
        const uint8_t *in_text = NULL;
        const uint8_t *out_text = NULL;

        if (targets == NULL)
                goto end_alloc;

        memset(targets, 0, num_jobs * sizeof(void *));

        if (auths == NULL)
                goto end_alloc;

        if (dir == IMB_DIR_ENCRYPT) {
                in_text = p_vec->pt;
                out_text = p_vec->ct;
        } else {
                in_text = p_vec->ct;
                out_text = p_vec->pt;
        }

        memset(padding, -1, sizeof(padding));

        for (i = 0; i < num_jobs; i++) {
                targets[i] = malloc(frame_len + (sizeof(padding) * 2));
                if (targets[i] == NULL)
                        goto end_alloc;
                memset(targets[i], -1, frame_len + (sizeof(padding) * 2));
                if (in_place) {
                        /* copy input text to the allocated buffer */
                        memcpy(targets[i] + sizeof(padding), in_text,
                               frame_len);
                        if (dir == IMB_DIR_ENCRYPT && p_vec->hash_length >= 14)
                                memset(targets[i] + sizeof(padding) +
                                       frame_len - tag_len, 0xff, tag_len);
                }
        }

        /* flush the scheduler */
        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL)
                ;

        for (i = 0; i < num_jobs; i++) {
                job = IMB_GET_NEXT_JOB(mb_mgr);
                job->cipher_direction = dir;
                job->chain_order = order;
                if (!in_place) {
                        job->dst = targets[i] + sizeof(padding);
                        job->src = in_text;
                } else {
                        job->dst = targets[i] + sizeof(padding) +
                                p_vec->cipher_offset;
                        job->src = targets[i] + sizeof(padding);
                }
                job->cipher_mode = IMB_CIPHER_DOCSIS_SEC_BPI;
                job->enc_keys = enc_keys;
                job->dec_keys = dec_keys;
                job->key_len_in_bytes = key_len;

                job->iv = p_vec->iv;
                job->iv_len_in_bytes = 16;
                job->cipher_start_src_offset_in_bytes = p_vec->cipher_offset;
                job->msg_len_to_cipher_in_bytes = p_vec->cipher_length;
                job->user_data = targets[i];
                job->user_data2 = (void *)((uint64_t)i);

                job->hash_alg = IMB_AUTH_DOCSIS_CRC32;
                job->hash_start_src_offset_in_bytes = p_vec->hash_offset;
                job->msg_len_to_hash_in_bytes = p_vec->hash_length;

                job->auth_tag_output = (uint8_t *) &auths[i];
                job->auth_tag_output_len_in_bytes = tag_len;
                job = IMB_SUBMIT_JOB(mb_mgr);
                if (job != NULL) {
                        jobs_rx++;
                        if (!docrc_job_ok(job, p_vec, job->user_data, padding,
                                          sizeof(padding), out_text))
                                goto end;
                }
        }

        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL) {
                jobs_rx++;
                if (!docrc_job_ok(job, p_vec, job->user_data, padding,
                                  sizeof(padding), out_text))
                        goto end;
        }

        if (jobs_rx != num_jobs) {
                printf("Expected %d jobs, received %d\n", num_jobs, jobs_rx);
                goto end;
        }
        ret = 0;

 end:
        while ((job = IMB_FLUSH_JOB(mb_mgr)) != NULL)
                ;
 end_alloc:

        if (targets != NULL) {
                for (i = 0; i < num_jobs; i++)
                        free(targets[i]);
                free(targets);
        }
        free(auths);

        return ret;
}

static void
test_docrc_vectors(struct IMB_MGR *mb_mgr,
                   struct test_suite_context *ctx128,
                   struct test_suite_context *ctx256,
                   const int vec_cnt,
                   const struct docsis_crc_vector *vec_tab,
                   const char *banner, const int num_jobs)
{
	int vect;
        DECLARE_ALIGNED(uint32_t enc_keys[15*4], 16);
        DECLARE_ALIGNED(uint32_t dec_keys[15*4], 16);

	printf("%s (N jobs = %d):\n", banner, num_jobs);
	for (vect = 0; vect < vec_cnt; vect++) {
                struct test_suite_context *ctx;
#ifdef DEBUG
		printf("[%d/%d] Standard vector\n",
                       vect + 1, vec_cnt);
#else
		printf(".");
#endif
                switch (vec_tab[vect].key_len) {
                case 16:
                        IMB_AES_KEYEXP_128(mb_mgr, vec_tab[vect].key, enc_keys,
                                           dec_keys);
                        ctx = ctx128;
                        break;
                case 32:
                default:
                        IMB_AES_KEYEXP_256(mb_mgr, vec_tab[vect].key, enc_keys,
                                           dec_keys);
                        ctx = ctx256;
                        break;
                }


                if (test_docrc_many(mb_mgr, enc_keys, dec_keys,
                                    &vec_tab[vect],
                                    IMB_DIR_ENCRYPT, IMB_ORDER_HASH_CIPHER, 1,
                                    num_jobs)) {
                        printf("error #%d encrypt\n", vect + 1);
                        test_suite_update(ctx, 0, 1);
                } else {
                        test_suite_update(ctx, 1, 0);
                }

                if (test_docrc_many(mb_mgr, enc_keys, dec_keys,
                                    &vec_tab[vect],
                                    IMB_DIR_DECRYPT, IMB_ORDER_CIPHER_HASH, 1,
                                    num_jobs)) {
                        printf("error #%d decrypt\n", vect + 1);
                        test_suite_update(ctx, 0, 1);
                } else {
                        test_suite_update(ctx, 1, 0);
                }

	}
	printf("\n");
}

static int
cfb128_validate_ok(const uint8_t *output, const uint8_t *in_text,
                   const size_t plen, const unsigned i, const unsigned is_enc,
                   const int in_place)
{
        if (memcmp(output, in_text, plen) != 0) {
                printf("\nAES-CFB128 standard test vector %d %s (%s): fail\n",
                       i + 1, (is_enc) ? "encrypt" : "decrypt",
                       (in_place) ? "in-place" : "out-of-place");
                return 0;
        }
#ifdef DEBUG
        printf("Standard test vector %u %s %s\n", i + 1,
               (in_place) ? "in-place" : "out-of-place",
               (is_enc) ? "encrypt" : "decrypt");
#else
	printf(".");
#endif

        return 1;
}


static int
cfb128_validate(struct IMB_MGR *mb_mgr)
{
        unsigned i;

        printf("AES-CFB128 standard test vectors:\n");
        for (i = 0; i < DIM(aes_cfb_128_tab); i++) {
                uint8_t output1[16];
                uint8_t output2[16];
                DECLARE_ALIGNED(uint32_t key[4], 16);
                DECLARE_ALIGNED(uint32_t keys_enc[11*4], 16);
                DECLARE_ALIGNED(uint32_t keys_dec[11*4], 16);

                memcpy(key, aes_cfb_128_tab[i].K, aes_cfb_128_tab[i].Klen);
                IMB_AES_KEYEXP_128(mb_mgr, key, keys_enc, keys_dec);

                /* Out of place */

                /* encrypt test */
                IMB_AES128_CFB_ONE(mb_mgr, output1, aes_cfb_128_tab[i].P,
                                   aes_cfb_128_tab[i].IV, keys_enc,
                                   aes_cfb_128_tab[i].Plen);
                if (!cfb128_validate_ok(output1, aes_cfb_128_tab[i].C,
                        aes_cfb_128_tab[i].Plen, i, 1, 0))
                        return 0;

                /* decrypt test */
                IMB_AES128_CFB_ONE(mb_mgr, output2, output1,
                                   aes_cfb_128_tab[i].IV, keys_enc,
                                   aes_cfb_128_tab[i].Plen);
                if (!cfb128_validate_ok(output2, aes_cfb_128_tab[i].P,
                        aes_cfb_128_tab[i].Plen, i, 0, 0))
                        return 0;

                /* In place */

                /* encrypt test */
                memcpy(output1, aes_cfb_128_tab[i].P, aes_cfb_128_tab[i].Plen);
                IMB_AES128_CFB_ONE(mb_mgr, output1, output1,
                                   aes_cfb_128_tab[i].IV, keys_enc,
                                   aes_cfb_128_tab[i].Plen);
                if (!cfb128_validate_ok(output1, aes_cfb_128_tab[i].C,
                        aes_cfb_128_tab[i].Plen, i, 1, 1))
                        return 0;

                /* decrypt test */
                memcpy(output1, aes_cfb_128_tab[i].C, aes_cfb_128_tab[i].Plen);
                IMB_AES128_CFB_ONE(mb_mgr, output1, output1,
                                   aes_cfb_128_tab[i].IV, keys_enc,
                                   aes_cfb_128_tab[i].Plen);
                if (!cfb128_validate_ok(output1, aes_cfb_128_tab[i].P,
                        aes_cfb_128_tab[i].Plen, i, 0, 1))
                        return 0;
        }
        printf("\n");
        return 1;
}

int
aes_test(struct IMB_MGR *mb_mgr)
{
        const int num_jobs_tab[] = {
                1, 3, 4, 5, 7, 8, 9, 15, 16, 17
        };
        unsigned i;
        int errors = 0;
        struct test_suite_context ctx128;
        struct test_suite_context ctx192;
        struct test_suite_context ctx256;

        test_suite_start(&ctx128, "AES-CBC-128");
        test_suite_start(&ctx192, "AES-CBC-192");
        test_suite_start(&ctx256, "AES-CBC-256");
        for (i = 0; i < DIM(num_jobs_tab); i++)
                test_aes_vectors(mb_mgr, &ctx128, &ctx192, &ctx256,
                                 DIM(aes_vectors),
                                 aes_vectors,
                                 "AES-CBC standard test vectors",
                                 IMB_CIPHER_CBC,
                                 num_jobs_tab[i]);
        errors += test_suite_end(&ctx128);
        errors += test_suite_end(&ctx192);
        errors += test_suite_end(&ctx256);

        test_suite_start(&ctx128, "DOCSIS-SEC-128");
        test_suite_start(&ctx256, "DOCSIS-SEC-256");
        for (i = 0; i < DIM(num_jobs_tab); i++)
                test_aes_vectors(mb_mgr, &ctx128, &ctx192, &ctx256,
                                 DIM(docsis_vectors),
                                 docsis_vectors,
                                 "AES-DOCSIS standard test vectors",
                                 IMB_CIPHER_DOCSIS_SEC_BPI,
                                 num_jobs_tab[i]);
        errors += test_suite_end(&ctx128);
        errors += test_suite_end(&ctx256);

        if (!cfb128_validate(mb_mgr))
                errors++;

        test_suite_start(&ctx128, "DOCSIS-SEC-128-CRC32");
        test_suite_start(&ctx256, "DOCSIS-SEC-256-CRC32");
        for (i = 0; i < DIM(num_jobs_tab); i++)
                test_docrc_vectors(mb_mgr, &ctx128, &ctx256,
                                   DIM(docsis_crc_tab),
                                   docsis_crc_tab,
                                   "AES-DOCSIS+CRC32 vectors",
                                   num_jobs_tab[i]);
        errors += test_suite_end(&ctx128);
        errors += test_suite_end(&ctx256);

        return errors;
}
