/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <unistd.h> /* sleep() */
#include <stdlib.h>
#include <string.h>

#include "cdw_dynamic_label.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"





/**
   \brief Create new dynamic label widget

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   \param parent - parent window
   \param field - ncurses field in parent form
   \param n_cols - width of the label
   \param content - initial contents of the label

   \return new widget on success
   \return NULL on errors
*/
CDW_DYNAMIC_LABEL *cdw_dynamic_label_new(WINDOW *parent, FIELD *field, int n_cols, const char *content)
{
	cdw_assert (parent, "ERROR: parent window can't be NULL\n");
	cdw_assert (content, "ERROR: initial content of the label can't be NULL\n");

	CDW_DYNAMIC_LABEL *label = (CDW_DYNAMIC_LABEL *) malloc(sizeof (CDW_DYNAMIC_LABEL));
	if (!label) {
		cdw_vdm ("ERROR: failed to malloc label widget\n");
		return (CDW_DYNAMIC_LABEL *) NULL;
	}

	label->content = strdup(content);
	if (!(label->content)) {
		cdw_vdm ("ERROR: failed to malloc content string\n");
		free(label);
		label = (CDW_DYNAMIC_LABEL *) NULL;
		return (CDW_DYNAMIC_LABEL *) NULL;
	}

	cdw_widget_init(&label->widget);
	label->widget.add_return_key = NULL;
	label->widget.type_id = CDW_WIDGET_ID_DYNAMIC_LABEL;

	label->parent = parent;
	label->field = field;
	label->n_cols = n_cols;

	int rv = set_field_buffer(label->field, 0, label->content);
	if (rv == E_OK) {
		return label;
	} else {
		cdw_dynamic_label_delete(&label);
		return (CDW_DYNAMIC_LABEL *) NULL;
	}
}





/**
   \brief Delete widget

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   Deallocate all memory associated with the widget. Set \p label to
   NULL.

   \param label - pointer to widget to delete
*/
void cdw_dynamic_label_delete(CDW_DYNAMIC_LABEL **label)
{
	cdw_assert (label, "ERROR: passing to the function a NULL pointer to a widget\n");

	if (!*label) {
		cdw_vdm ("WARNING: passing NULL label to the function\n");
		return;
	}

	if ((*label)->content) {
		free((*label)->content);
		(*label)->content = (char *) NULL;
	}

	free(*label);
	*label = (CDW_DYNAMIC_LABEL *) NULL;
	return;
}





/**
   \brief Set new contents of a dynamic label widget

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   \param label - dynamic label widget
   \param content - new content of dynamic label widget

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_dynamic_label_set_new(CDW_DYNAMIC_LABEL *label, const char *content)
{
	cdw_assert (content, "ERROR: can't set new NULL content\n");

	cdw_rv_t crv = cdw_string_set(&(label->content), content);
	if (crv == CDW_OK) {
		int rv = set_field_buffer(label->field, 0, label->content);
		if (rv == E_OK) {
			return CDW_OK;
		}
	}

	return CDW_ERROR;
}




/**
   \brief Set new, temporary contents of dynamic label widget

   \date Function's top-level comment reviewed on 2014-03-30
   \date Function's body reviewed on 2014-03-30

   Display string from \p temporary_content for \p seconds seconds
   using \p colors color scheme. After \p seconds seconds display
   again the original contents of the label using the original colors
   scheme.

   \param label - dynamic label widget
   \param temporary_content - temporary content to be set
   \param seconds - period of time in which to display the temporary contents
   \param colors - fore/background colors scheme to use for displaying the temporary contents

   \return CDW_OK
*/
cdw_rv_t cdw_dynamic_label_set_new_temporary(CDW_DYNAMIC_LABEL *label, const char *temporary_content, unsigned int seconds, cdw_colors_t colors)
{
	cdw_assert (temporary_content, "ERROR: can't set new NULL content\n");

	/* Display the temporary content... */
	set_field_fore(label->field, COLOR_PAIR(colors));
	set_field_buffer(label->field, 0, temporary_content);
	wrefresh(label->parent);

	/* ... but only for X seconds.  During this time input from
	   user is blocked. */
	sleep(seconds);

	/* Restore original string with regular colors. */
	set_field_fore(label->field, COLOR_PAIR(CDW_COLORS_DIALOG));
	set_field_buffer(label->field, 0, label->content);
	wrefresh(label->parent);

	return CDW_OK;
}
