
/* Convert the ICC V4.4 Reference Medium Gamut to Argyll Lab Gamut */
/* Copyright 2005, 2024 by Graeme W. Gill */
/* All rights reserved. */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <string.h>
#include <math.h>
#include "aconfig.h"
#include "numlib.h"
#include "icc.h"
#include "cgats.h"
#include "xicc.h"
#include "xcam.h"
#include "gamut.h"

/* Gamut surface points in LCh space */
/* + black and white points */
#define NPOINTS (684+2+3)

double points[NPOINTS][3] = {
	{ 100,	0,	0 },			/* White */
	{ 3.1373,	0,	0 },		/* Black */
	{ 98.75, 30.75, 90 },		/* Three extra values along yellow ridge */
	{ 97.5, 61.5, 90 },
	{ 96.25, 92.25, 90 },
	{ 5,	11,	0 },
	{ 10,	26,	0 },
	{ 15,	39,	0 },
	{ 20,	52,	0 },
	{ 25,	64,	0 },
	{ 30,	74,	0 },
	{ 35,	83,	0 },
	{ 40,	91,	0 },
	{ 45,	92,	0 },
	{ 50,	91,	0 },
	{ 55,	87,	0 },
	{ 60,	82,	0 },
	{ 65,	75,	0 },
	{ 70,	67,	0 },
	{ 75,	57,	0 },
	{ 80,	47,	0 },
	{ 85,	37,	0 },
	{ 90,	25,	0 },
	{ 95,	13,	0 },
	{ 5,	10,	10 },
	{ 10,	24,	10 },
	{ 15,	38,	10 },
	{ 20,	50,	10 },
	{ 25,	62,	10 },
	{ 30,	73,	10 },
	{ 35,	82,	10 },
	{ 40,	90,	10 },
	{ 45,	92,	10 },
	{ 50,	91,	10 },
	{ 55,	87,	10 },
	{ 60,	82,	10 },
	{ 65,	75,	10 },
	{ 70,	67,	10 },
	{ 75,	58,	10 },
	{ 80,	48,	10 },
	{ 85,	37,	10 },
	{ 90,	26,	10 },
	{ 95,	13,	10 },
	{ 5,	10,	20 },
	{ 10,	23,	20 },
	{ 15,	37,	20 },
	{ 20,	50,	20 },
	{ 25,	62,	20 },
	{ 30,	73,	20 },
	{ 35,	84,	20 },
	{ 40,	93,	20 },
	{ 45,	94,	20 },
	{ 50,	94,	20 },
	{ 55,	90,	20 },
	{ 60,	85,	20 },
	{ 65,	78,	20 },
	{ 70,	70,	20 },
	{ 75,	60,	20 },
	{ 80,	50,	20 },
	{ 85,	39,	20 },
	{ 90,	27,	20 },
	{ 95,	14,	20 },
	{ 5,	9,	30 },
	{ 10,	22,	30 },
	{ 15,	35,	30 },
	{ 20,	48,	30 },
	{ 25,	61,	30 },
	{ 30,	74,	30 },
	{ 35,	86,	30 },
	{ 40,	98,	30 },
	{ 45,	100,	30 },
	{ 50,	101,	30 },
	{ 55,	96,	30 },
	{ 60,	90,	30 },
	{ 65,	83,	30 },
	{ 70,	75,	30 },
	{ 75,	65,	30 },
	{ 80,	54,	30 },
	{ 85,	42,	30 },
	{ 90,	30,	30 },
	{ 95,	15,	30 },
	{ 5,	8,	40 },
	{ 10,	21,	40 },
	{ 15,	34,	40 },
	{ 20,	47,	40 },
	{ 25,	60,	40 },
	{ 30,	73,	40 },
	{ 35,	83,	40 },
	{ 40,	93,	40 },
	{ 45,	97,	40 },
	{ 50,	101,	40 },
	{ 55,	99,	40 },
	{ 60,	97,	40 },
	{ 65,	90,	40 },
	{ 70,	83,	40 },
	{ 75,	73,	40 },
	{ 80,	61,	40 },
	{ 85,	47,	40 },
	{ 90,	34,	40 },
	{ 95,	17,	40 },
	{ 5,	8,	50 },
	{ 10,	20,	50 },
	{ 15,	32,	50 },
	{ 20,	43,	50 },
	{ 25,	55,	50 },
	{ 30,	66,	50 },
	{ 35,	77,	50 },
	{ 40,	88,	50 },
	{ 45,	95,	50 },
	{ 50,	99,	50 },
	{ 55,	101,	50 },
	{ 60,	100,	50 },
	{ 65,	98,	50 },
	{ 70,	92,	50 },
	{ 75,	85,	50 },
	{ 80,	72,	50 },
	{ 85,	56,	50 },
	{ 90,	40,	50 },
	{ 95,	20,	50 },
	{ 5,	7,	60 },
	{ 10,	17,	60 },
	{ 15,	27,	60 },
	{ 20,	37,	60 },
	{ 25,	47,	60 },
	{ 30,	57,	60 },
	{ 35,	67,	60 },
	{ 40,	76,	60 },
	{ 45,	84,	60 },
	{ 50,	91,	60 },
	{ 55,	96,	60 },
	{ 60,	100,	60 },
	{ 65,	102,	60 },
	{ 70,	103,	60 },
	{ 75,	98,	60 },
	{ 80,	90,	60 },
	{ 85,	72,	60 },
	{ 90,	51,	60 },
	{ 95,	26,	60 },
	{ 5,	6,	70 },
	{ 10,	16,	70 },
	{ 15,	25,	70 },
	{ 20,	34,	70 },
	{ 25,	43,	70 },
	{ 30,	52,	70 },
	{ 35,	60,	70 },
	{ 40,	68,	70 },
	{ 45,	76,	70 },
	{ 50,	83,	70 },
	{ 55,	90,	70 },
	{ 60,	96,	70 },
	{ 65,	100,	70 },
	{ 70,	104,	70 },
	{ 75,	107,	70 },
	{ 80,	109,	70 },
	{ 85,	100,	70 },
	{ 90,	74,	70 },
	{ 95,	37,	70 },
	{ 5,	6,	80 },
	{ 10,	15,	80 },
	{ 15,	23,	80 },
	{ 20,	32,	80 },
	{ 25,	40,	80 },
	{ 30,	48,	80 },
	{ 35,	57,	80 },
	{ 40,	64,	80 },
	{ 45,	71,	80 },
	{ 50,	78,	80 },
	{ 55,	85,	80 },
	{ 60,	91,	80 },
	{ 65,	97,	80 },
	{ 70,	103,	80 },
	{ 75,	107,	80 },
	{ 80,	110,	80 },
	{ 85,	113,	80 },
	{ 90,	110,	80 },
	{ 95,	70,	80 },
	{ 5,	6,	90 },
	{ 10,	14,	90 },
	{ 15,	22,	90 },
	{ 20,	30,	90 },
	{ 25,	39,	90 },
	{ 30,	47,	90 },
	{ 35,	55,	90 },
	{ 40,	62,	90 },
	{ 45,	68,	90 },
	{ 50,	75,	90 },
	{ 55,	82,	90 },
	{ 60,	88,	90 },
	{ 65,	95,	90 },
	{ 70,	101,	90 },
	{ 75,	106,	90 },
	{ 80,	112,	90 },
	{ 85,	117,	90 },
	{ 90,	120,	90 },
	{ 95,	123,	90 },
	{ 5,	6,	100 },
	{ 10,	14,	100 },
	{ 15,	22,	100 },
	{ 20,	30,	100 },
	{ 25,	38,	100 },
	{ 30,	46,	100 },
	{ 35,	54,	100 },
	{ 40,	61,	100 },
	{ 45,	68,	100 },
	{ 50,	74,	100 },
	{ 55,	81,	100 },
	{ 60,	88,	100 },
	{ 65,	94,	100 },
	{ 70,	100,	100 },
	{ 75,	106,	100 },
	{ 80,	109,	100 },
	{ 85,	112,	100 },
	{ 90,	112,	100 },
	{ 95,	92,	100 },
	{ 5,	6,	110 },
	{ 10,	14,	110 },
	{ 15,	22,	110 },
	{ 20,	31,	110 },
	{ 25,	39,	110 },
	{ 30,	47,	110 },
	{ 35,	55,	110 },
	{ 40,	63,	110 },
	{ 45,	69,	110 },
	{ 50,	76,	110 },
	{ 55,	83,	110 },
	{ 60,	89,	110 },
	{ 65,	96,	110 },
	{ 70,	100,	110 },
	{ 75,	103,	110 },
	{ 80,	106,	110 },
	{ 85,	107,	110 },
	{ 90,	102,	110 },
	{ 95,	75,	110 },
	{ 5,	6,	120 },
	{ 10,	15,	120 },
	{ 15,	24,	120 },
	{ 20,	32,	120 },
	{ 25,	41,	120 },
	{ 30,	49,	120 },
	{ 35,	58,	120 },
	{ 40,	66,	120 },
	{ 45,	73,	120 },
	{ 50,	80,	120 },
	{ 55,	87,	120 },
	{ 60,	93,	120 },
	{ 65,	98,	120 },
	{ 70,	101,	120 },
	{ 75,	102,	120 },
	{ 80,	99,	120 },
	{ 85,	91,	120 },
	{ 90,	73,	120 },
	{ 95,	50,	120 },
	{ 5,	6,	130 },
	{ 10,	16,	130 },
	{ 15,	25,	130 },
	{ 20,	35,	130 },
	{ 25,	44,	130 },
	{ 30,	54,	130 },
	{ 35,	63,	130 },
	{ 40,	72,	130 },
	{ 45,	80,	130 },
	{ 50,	87,	130 },
	{ 55,	93,	130 },
	{ 60,	97,	130 },
	{ 65,	101,	130 },
	{ 70,	99,	130 },
	{ 75,	94,	130 },
	{ 80,	86,	130 },
	{ 85,	73,	130 },
	{ 90,	56,	130 },
	{ 95,	34,	130 },
	{ 5,	7,	140 },
	{ 10,	18,	140 },
	{ 15,	28,	140 },
	{ 20,	38,	140 },
	{ 25,	48,	140 },
	{ 30,	57,	140 },
	{ 35,	67,	140 },
	{ 40,	77,	140 },
	{ 45,	86,	140 },
	{ 50,	95,	140 },
	{ 55,	98,	140 },
	{ 60,	101,	140 },
	{ 65,	97,	140 },
	{ 70,	93,	140 },
	{ 75,	85,	140 },
	{ 80,	75,	140 },
	{ 85,	61,	140 },
	{ 90,	44,	140 },
	{ 95,	26,	140 },
	{ 5,	7,	150 },
	{ 10,	19,	150 },
	{ 15,	30,	150 },
	{ 20,	40,	150 },
	{ 25,	51,	150 },
	{ 30,	62,	150 },
	{ 35,	72,	150 },
	{ 40,	83,	150 },
	{ 45,	92,	150 },
	{ 50,	97,	150 },
	{ 55,	99,	150 },
	{ 60,	96,	150 },
	{ 65,	91,	150 },
	{ 70,	85,	150 },
	{ 75,	76,	150 },
	{ 80,	66,	150 },
	{ 85,	52,	150 },
	{ 90,	37,	150 },
	{ 95,	22,	150 },
	{ 5,	7,	160 },
	{ 10,	20,	160 },
	{ 15,	32,	160 },
	{ 20,	44,	160 },
	{ 25,	56,	160 },
	{ 30,	68,	160 },
	{ 35,	80,	160 },
	{ 40,	92,	160 },
	{ 45,	96,	160 },
	{ 50,	99,	160 },
	{ 55,	97,	160 },
	{ 60,	92,	160 },
	{ 65,	87,	160 },
	{ 70,	79,	160 },
	{ 75,	70,	160 },
	{ 80,	59,	160 },
	{ 85,	46,	160 },
	{ 90,	33,	160 },
	{ 95,	19,	160 },
	{ 5,	8,	170 },
	{ 10,	20,	170 },
	{ 15,	32,	170 },
	{ 20,	43,	170 },
	{ 25,	53,	170 },
	{ 30,	64,	170 },
	{ 35,	75,	170 },
	{ 40,	85,	170 },
	{ 45,	91,	170 },
	{ 50,	96,	170 },
	{ 55,	93,	170 },
	{ 60,	89,	170 },
	{ 65,	82,	170 },
	{ 70,	75,	170 },
	{ 75,	65,	170 },
	{ 80,	55,	170 },
	{ 85,	42,	170 },
	{ 90,	30,	170 },
	{ 95,	17,	170 },
	{ 5,	8,	180 },
	{ 10,	20,	180 },
	{ 15,	31,	180 },
	{ 20,	41,	180 },
	{ 25,	52,	180 },
	{ 30,	62,	180 },
	{ 35,	72,	180 },
	{ 40,	81,	180 },
	{ 45,	87,	180 },
	{ 50,	92,	180 },
	{ 55,	90,	180 },
	{ 60,	86,	180 },
	{ 65,	79,	180 },
	{ 70,	71,	180 },
	{ 75,	61,	180 },
	{ 80,	52,	180 },
	{ 85,	40,	180 },
	{ 90,	28,	180 },
	{ 95,	15,	180 },
	{ 5,	8,	190 },
	{ 10,	20,	190 },
	{ 15,	30,	190 },
	{ 20,	40,	190 },
	{ 25,	50,	190 },
	{ 30,	60,	190 },
	{ 35,	68,	190 },
	{ 40,	76,	190 },
	{ 45,	82,	190 },
	{ 50,	87,	190 },
	{ 55,	85,	190 },
	{ 60,	82,	190 },
	{ 65,	76,	190 },
	{ 70,	69,	190 },
	{ 75,	60,	190 },
	{ 80,	50,	190 },
	{ 85,	39,	190 },
	{ 90,	27,	190 },
	{ 95,	14,	190 },
	{ 5,	8,	200 },
	{ 10,	20,	200 },
	{ 15,	30,	200 },
	{ 20,	38,	200 },
	{ 25,	47,	200 },
	{ 30,	56,	200 },
	{ 35,	63,	200 },
	{ 40,	70,	200 },
	{ 45,	76,	200 },
	{ 50,	82,	200 },
	{ 55,	81,	200 },
	{ 60,	77,	200 },
	{ 65,	72,	200 },
	{ 70,	66,	200 },
	{ 75,	58,	200 },
	{ 80,	49,	200 },
	{ 85,	38,	200 },
	{ 90,	27,	200 },
	{ 95,	14,	200 },
	{ 5,	8,	210 },
	{ 10,	20,	210 },
	{ 15,	29,	210 },
	{ 20,	37,	210 },
	{ 25,	46,	210 },
	{ 30,	53,	210 },
	{ 35,	60,	210 },
	{ 40,	66,	210 },
	{ 45,	73,	210 },
	{ 50,	79,	210 },
	{ 55,	80,	210 },
	{ 60,	75,	210 },
	{ 65,	70,	210 },
	{ 70,	64,	210 },
	{ 75,	57,	210 },
	{ 80,	49,	210 },
	{ 85,	38,	210 },
	{ 90,	27,	210 },
	{ 95,	14,	210 },
	{ 5,	8,	220 },
	{ 10,	20,	220 },
	{ 15,	29,	220 },
	{ 20,	37,	220 },
	{ 25,	45,	220 },
	{ 30,	52,	220 },
	{ 35,	59,	220 },
	{ 40,	65,	220 },
	{ 45,	71,	220 },
	{ 50,	76,	220 },
	{ 55,	75,	220 },
	{ 60,	72,	220 },
	{ 65,	68,	220 },
	{ 70,	63,	220 },
	{ 75,	56,	220 },
	{ 80,	48,	220 },
	{ 85,	38,	220 },
	{ 90,	27,	220 },
	{ 95,	14,	220 },
	{ 5,	9,	230 },
	{ 10,	20,	230 },
	{ 15,	29,	230 },
	{ 20,	38,	230 },
	{ 25,	46,	230 },
	{ 30,	53,	230 },
	{ 35,	59,	230 },
	{ 40,	65,	230 },
	{ 45,	70,	230 },
	{ 50,	75,	230 },
	{ 55,	73,	230 },
	{ 60,	71,	230 },
	{ 65,	66,	230 },
	{ 70,	61,	230 },
	{ 75,	54,	230 },
	{ 80,	46,	230 },
	{ 85,	36,	230 },
	{ 90,	26,	230 },
	{ 95,	13,	230 },
	{ 5,	10,	240 },
	{ 10,	22,	240 },
	{ 15,	31,	240 },
	{ 20,	40,	240 },
	{ 25,	48,	240 },
	{ 30,	55,	240 },
	{ 35,	61,	240 },
	{ 40,	67,	240 },
	{ 45,	71,	240 },
	{ 50,	74,	240 },
	{ 55,	70,	240 },
	{ 60,	66,	240 },
	{ 65,	61,	240 },
	{ 70,	56,	240 },
	{ 75,	49,	240 },
	{ 80,	41,	240 },
	{ 85,	32,	240 },
	{ 90,	23,	240 },
	{ 95,	12,	240 },
	{ 5,	11,	250 },
	{ 10,	24,	250 },
	{ 15,	34,	250 },
	{ 20,	43,	250 },
	{ 25,	51,	250 },
	{ 30,	59,	250 },
	{ 35,	65,	250 },
	{ 40,	70,	250 },
	{ 45,	73,	250 },
	{ 50,	71,	250 },
	{ 55,	68,	250 },
	{ 60,	63,	250 },
	{ 65,	58,	250 },
	{ 70,	52,	250 },
	{ 75,	45,	250 },
	{ 80,	38,	250 },
	{ 85,	30,	250 },
	{ 90,	21,	250 },
	{ 95,	11,	250 },
	{ 5,	14,	260 },
	{ 10,	27,	260 },
	{ 15,	38,	260 },
	{ 20,	48,	260 },
	{ 25,	57,	260 },
	{ 30,	64,	260 },
	{ 35,	69,	260 },
	{ 40,	73,	260 },
	{ 45,	73,	260 },
	{ 50,	70,	260 },
	{ 55,	66,	260 },
	{ 60,	61,	260 },
	{ 65,	56,	260 },
	{ 70,	50,	260 },
	{ 75,	43,	260 },
	{ 80,	35,	260 },
	{ 85,	28,	260 },
	{ 90,	20,	260 },
	{ 95,	10,	260 },
	{ 5,	17,	270 },
	{ 10,	32,	270 },
	{ 15,	45,	270 },
	{ 20,	55,	270 },
	{ 25,	65,	270 },
	{ 30,	70,	270 },
	{ 35,	75,	270 },
	{ 40,	75,	270 },
	{ 45,	73,	270 },
	{ 50,	70,	270 },
	{ 55,	66,	270 },
	{ 60,	61,	270 },
	{ 65,	55,	270 },
	{ 70,	49,	270 },
	{ 75,	42,	270 },
	{ 80,	34,	270 },
	{ 85,	27,	270 },
	{ 90,	19,	270 },
	{ 95,	10,	270 },
	{ 5,	21,	280 },
	{ 10,	42,	280 },
	{ 15,	55,	280 },
	{ 20,	68,	280 },
	{ 25,	75,	280 },
	{ 30,	81,	280 },
	{ 35,	80,	280 },
	{ 40,	79,	280 },
	{ 45,	76,	280 },
	{ 50,	72,	280 },
	{ 55,	67,	280 },
	{ 60,	61,	280 },
	{ 65,	55,	280 },
	{ 70,	49,	280 },
	{ 75,	41,	280 },
	{ 80,	34,	280 },
	{ 85,	26,	280 },
	{ 90,	18,	280 },
	{ 95,	9,	280 },
	{ 5,	26,	290 },
	{ 10,	52,	290 },
	{ 15,	68,	290 },
	{ 20,	83,	290 },
	{ 25,	86,	290 },
	{ 30,	89,	290 },
	{ 35,	87,	290 },
	{ 40,	84,	290 },
	{ 45,	80,	290 },
	{ 50,	75,	290 },
	{ 55,	69,	290 },
	{ 60,	63,	290 },
	{ 65,	57,	290 },
	{ 70,	50,	290 },
	{ 75,	42,	290 },
	{ 80,	35,	290 },
	{ 85,	27,	290 },
	{ 90,	18,	290 },
	{ 95,	10,	290 },
	{ 5,	25,	300 },
	{ 10,	69,	300 },
	{ 15,	82,	300 },
	{ 20,	95,	300 },
	{ 25,	94,	300 },
	{ 30,	93,	300 },
	{ 35,	91,	300 },
	{ 40,	88,	300 },
	{ 45,	85,	300 },
	{ 50,	79,	300 },
	{ 55,	73,	300 },
	{ 60,	66,	300 },
	{ 65,	59,	300 },
	{ 70,	52,	300 },
	{ 75,	44,	300 },
	{ 80,	36,	300 },
	{ 85,	28,	300 },
	{ 90,	19,	300 },
	{ 95,	10,	300 },
	{ 5,	21,	310 },
	{ 10,	51,	310 },
	{ 15,	74,	310 },
	{ 20,	91,	310 },
	{ 25,	97,	310 },
	{ 30,	100,	310 },
	{ 35,	98,	310 },
	{ 40,	95,	310 },
	{ 45,	90,	310 },
	{ 50,	84,	310 },
	{ 55,	77,	310 },
	{ 60,	70,	310 },
	{ 65,	63,	310 },
	{ 70,	55,	310 },
	{ 75,	47,	310 },
	{ 80,	39,	310 },
	{ 85,	30,	310 },
	{ 90,	20,	310 },
	{ 95,	10,	310 },
	{ 5,	18,	320 },
	{ 10,	41,	320 },
	{ 15,	62,	320 },
	{ 20,	79,	320 },
	{ 25,	91,	320 },
	{ 30,	102,	320 },
	{ 35,	101,	320 },
	{ 40,	98,	320 },
	{ 45,	95,	320 },
	{ 50,	89,	320 },
	{ 55,	83,	320 },
	{ 60,	76,	320 },
	{ 65,	68,	320 },
	{ 70,	60,	320 },
	{ 75,	51,	320 },
	{ 80,	42,	320 },
	{ 85,	32,	320 },
	{ 90,	22,	320 },
	{ 95,	11,	320 },
	{ 5,	16,	330 },
	{ 10,	35,	330 },
	{ 15,	53,	330 },
	{ 20,	71,	330 },
	{ 25,	82,	330 },
	{ 30,	91,	330 },
	{ 35,	100,	330 },
	{ 40,	104,	330 },
	{ 45,	102,	330 },
	{ 50,	98,	330 },
	{ 55,	91,	330 },
	{ 60,	84,	330 },
	{ 65,	76,	330 },
	{ 70,	67,	330 },
	{ 75,	57,	330 },
	{ 80,	47,	330 },
	{ 85,	36,	330 },
	{ 90,	24,	330 },
	{ 95,	12,	330 },
	{ 5,	14,	340 },
	{ 10,	31,	340 },
	{ 15,	46,	340 },
	{ 20,	61,	340 },
	{ 25,	73,	340 },
	{ 30,	83,	340 },
	{ 35,	92,	340 },
	{ 40,	101,	340 },
	{ 45,	103,	340 },
	{ 50,	99,	340 },
	{ 55,	95,	340 },
	{ 60,	89,	340 },
	{ 65,	80,	340 },
	{ 70,	71,	340 },
	{ 75,	61,	340 },
	{ 80,	50,	340 },
	{ 85,	38,	340 },
	{ 90,	26,	340 },
	{ 95,	13,	340 },
	{ 5,	12,	350 },
	{ 10,	28,	350 },
	{ 15,	42,	350 },
	{ 20,	55,	350 },
	{ 25,	68,	350 },
	{ 30,	77,	350 },
	{ 35,	86,	350 },
	{ 40,	94,	350 },
	{ 45,	96,	350 },
	{ 50,	93,	350 },
	{ 55,	90,	350 },
	{ 60,	85,	350 },
	{ 65,	77,	350 },
	{ 70,	68,	350 },
	{ 75,	58,	350 },
	{ 80,	48,	350 },
	{ 85,	37,	350 },
	{ 90,	25,	350 },
	{ 95,	13,	350 },
};

/* selected "cusp" points in LCh */
double cpoints[6][3] = {
	{ 41,	98,		29 },		/* Red */
	{ 95,	123,	90 },		/* Yellow */
	{ 60,	100,	140 },		/* Green */
	{ 50,	76,	    220 },		/* Cyan */ 
	{ 21,	95,	    300 },		/* Blue */
	{ 42,	102,	340 },		/* Megenta */
};

/* Our idealised cusp point angles for Lab in degrees: */ 
/* 36.0,		Red */
/* 101.0,		Yellow */
/* 149.0,		Green */
/* 225.0,		Cyan */
/* 300.0,		Blue */
/* 337.0		Magenta */

#define GAMRES 1.0		/* Default surface resolution */

void usage(char *diag) {
	fprintf(stderr,"Generate Reference Medium Gamut Version %s\n",ARGYLL_VERSION_STR);
	fprintf(stderr,"Author: Graeme W. Gill, licensed under the AGPL Version 3\n");
	fprintf(stderr,"usage: GenRMGam [options] [outputname]\n");
	if (diag != NULL)
		fprintf(stderr,"Diagnostic: %s\n",diag);
	fprintf(stderr," -v                Verbose\n");
	fprintf(stderr," -l                L*a*b* output rather than Jab ISO-3664 P2\n");
	fprintf(stderr," outputname        output name rather than default RefMediumGamut.gam\n");
	fprintf(stderr,"\n");
	exit(1);
}

int
main(int argc, char *argv[]) {
	int fa, nfa;				/* argument we're looking at */
	int verb = 0;
	int dolab = 0;
	icxcam *cam = NULL;					/* Used for Jab conversion */
	char out_name[MAXNAMEL+1];			/* VRML output file */
	double gamres = GAMRES;				/* Surface resolution */
	gamut *gam;
	double wp[3], bp[3];
	int i;

	error_program = "GenRMGam";

	/* Process the arguments */
	for(fa = 1;fa < argc;fa++) {
		nfa = fa;					/* skip to nfa if next argument is used */
		if (argv[fa][0] == '-')	{	/* Look for any flags */
			char *na = NULL;		/* next argument after flag, null if none */

			if (argv[fa][2] != '\000')
				na = &argv[fa][2];		/* next is directly after flag */
			else {
				if ((fa+1) < argc) {
					if (argv[fa+1][0] != '-') {
						nfa = fa + 1;
						na = argv[nfa];		/* next is seperate non-flag argument */
					}
				}
			}

			if (argv[fa][1] == '?')
				usage(NULL);

			/* Verbosity */
			else if (argv[fa][1] == 'v') {
				verb = 1;
			}

			/* Lab */
			else if (argv[fa][1] == 'l') {
				dolab = 1;
			}

			else 
				usage("Unknown flag");
		} else
			break;
	}

	if (fa >= argc || argv[fa][0] == '-') {
		strcpy(out_name, "RefMediumGamut.gam");
	} else {
		strncpy(out_name,argv[1],MAXNAMEL); out_name[MAXNAMEL] = '\000';
	}

	{
		double *wp = icmD50_ary3;
		icxViewCond vc;

		/* ISO-3664 P2 */
		if (xicc_enum_viewcond(NULL, &vc, -2, "pp", 0, wp) == -999)
		    error("Urecognised Enumerated Viewing conditions '%s'","pp");

		if (verb)
			xicc_dump_viewcond(&vc);

		if ((cam = new_icxcam(cam_CIECAM02)) == NULL)
		    error("new_icxcam failed");

		cam->set_view_vc(cam, &vc);

//		if (verb)
//			cam->dump(cam);
	}

	/* Creat a gamut surface */
	gam = new_gamut(gamres, cam != NULL, 0);

	/* For all the supplied surface points, */
	/* add them to the gamut */
	for (i = 0; i < NPOINTS; i++) {
		double lab[3];

		icmLCh2Lab(lab, points[i]);
//printf("~1 LCh %f %f %f -> Lab %f %f %f\n", points[i][0], points[i][1], points[i][2], lab[0], lab[1], lab[2]);

		/* Convert to appearance space */
		if (cam != NULL) {
			icmLab2XYZ(&icmD50, lab, lab);
			cam->XYZ_to_cam(cam, lab, lab);
//printf("~1 Jab %f %f %f\n", lab[0], lab[1], lab[2]);
		}

		gam->expand(gam, lab);
	}

	/* Add selected cusp points to set cusp values */
	gam->setcusps(gam, 0, NULL);
	for (i = 0; i < 6; i++) {
		double lab[3];

		icmLCh2Lab(lab, cpoints[i]);
//printf("~1 LCh %f %f %f -> Lab %f %f %f\n", points[i][0], points[i][1], points[i][2], lab[0], lab[1], lab[2]);
		/* Convert to appearance space */
		if (cam != NULL) {
			icmLab2XYZ(&icmD50, lab, lab);
			cam->XYZ_to_cam(cam, lab, lab);
		}

		gam->setcusps(gam, 1, lab);
	}
	gam->setcusps(gam, 2, NULL);

	/* Add colorspace white and black points so that this gamut can substitute */
	/* for source colorspace gamut */
	wp[0] = 100.0;
	wp[1] = 0.0;
	wp[2] = 0.0;
	bp[0] = 0.0;
	bp[1] = 0.0;
	bp[2] = 0.0;
	
	gam->setwb(gam, wp, bp, NULL);

	if (gam->write_gam(gam, out_name))
		error ("write gamut failed on '%s'",out_name);

	gam->del(gam);

	return 0;
}

