#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2025 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Diablo 2 expansions:
# - Lord of Destriction
# send your bug reports to vv221@dotslashplay.it
###

script_version=20250628.4

PLAYIT_COMPATIBILITY_LEVEL='2.32'

GAME_ID='diablo-2'
GAME_NAME='Diablo II'

EXPANSION_ID='lord-of-destruction'
EXPANSION_NAME='Lord of Destruction'

ARCHIVE_BASE_EN_0_NAME='Installer Tome.mpq'
ARCHIVE_BASE_EN_0_MD5='fb7bdc951003b8def5332f73060cd464'
ARCHIVE_BASE_EN_0_SIZE='489332'
ARCHIVE_BASE_EN_0_VERSION='1.14b-blizzard'
ARCHIVE_BASE_EN_0_URL='https://eu.battle.net/download/getLegacy?os=win&locale=enUS&product=D2XP'

ARCHIVE_BASE_FR_0_NAME='Installer Tome.mpq'
ARCHIVE_BASE_FR_0_MD5='5ff92ff2fc1c1eb5c81631b6ff56c804'
ARCHIVE_BASE_FR_0_SIZE='605156'
ARCHIVE_BASE_FR_0_VERSION='1.14b-blizzard'
ARCHIVE_BASE_FR_0_URL='https://eu.battle.net/download/getLegacy?os=win&locale=frFR&product=D2XP'

CONTENT_PATH_DEFAULT='.'
CONTENT_GAME_MAIN_PATH='Common'
CONTENT_GAME_MAIN_FILES='
d2exp.mpq
d2xtalk.mpq
d2xvideo.mpq'
CONTENT_GAME_DATA_MUSIC_PATH='Common'
CONTENT_GAME_DATA_MUSIC_FILES='
d2xmusic.mpq'
CONTENT_GAME_DATA_SHARED_PATH='Common'
CONTENT_GAME_DATA_SHARED_FILES='
d2char.mpq
patch_d2.mpq'

APP_MAIN_ICON='PC/Diablo II.exe'

PACKAGES_LIST='
PKG_MAIN
PKG_DATA_MUSIC
PKG_DATA_SHARED'

PKG_PARENT_ID="$GAME_ID"

PKG_MAIN_ID="${GAME_ID}-${EXPANSION_ID}"
PKG_MAIN_ID_EN="${PKG_MAIN_ID}-en"
PKG_MAIN_ID_FR="${PKG_MAIN_ID}-fr"
PKG_MAIN_PROVIDES="
$PKG_MAIN_ID"
PKG_MAIN_DEPENDENCIES_SIBLINGS='
PKG_PARENT
PKG_DATA_MUSIC
PKG_DATA_SHARED'

PKG_DATA_MUSIC_ID="${GAME_ID}-${EXPANSION_ID}-data-music"
PKG_DATA_MUSIC_DESCRIPTION='music'

PKG_DATA_SHARED_ID="${GAME_ID}-${EXPANSION_ID}-data-shared"
PKG_DATA_SHARED_PROVIDES="
$PKG_DATA_SHARED_ID
${GAME_ID}-data-shared"
PKG_DATA_SHARED_DESCRIPTION='data shared between the base game and the expansion'

# Set the list of requirements to extract the archive contents.

REQUIREMENTS_LIST="${REQUIREMENTS_LIST:-}
smpq"

# Load common functions

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run the default initialization actions

initialization_default "$@"

# Check for the presence of the CD key files.

## During a regular install of Lord of Destruction, the CD key is stored encrypted into d2char.mpq,
## in files under the "data/global/sfx/cursor" and "data/global/chars/am/cof" directories.
## In order to build a package that includes the CD key (required, or the game will not launch),
## you need to provide the following files in addition to the installer .mpq:
## - curindx.wav
## - amblxbow.cof
##
## They can be extracted from a regular Diablo 2 install with:
## smpq --extract d2char.mpq data/global/sfx/cursor/curindx.wav data/global/chars/am/cof/amblxbow.cof
ARCHIVE_REQUIRED_CDKEY_1_NAME='curindx.wav'
ARCHIVE_REQUIRED_CDKEY_2_NAME='amblxbow.cof'
archive_initialize_required \
	'ARCHIVE_CDKEY_1' \
	'ARCHIVE_REQUIRED_CDKEY_1'
archive_initialize_required \
	'ARCHIVE_CDKEY_2' \
	'ARCHIVE_REQUIRED_CDKEY_2'

## In addition to these files, the d2char.mpq archive from Diablo 2 base game is required.
## If Diablo 2 is installed on the current system, a symbolic link to the game file is enough.
ARCHIVE_REQUIRED_D2CHAR_NAME='d2char.mpq'
archive_initialize_required \
	'ARCHIVE_D2CHAR' \
	'ARCHIVE_REQUIRED_D2CHAR'

# Extract game data

archive_path=$(archive_path "$(current_archive)")
archive_path_cdkey_1=$(archive_path 'ARCHIVE_CDKEY_1')
archive_path_cdkey_2=$(archive_path 'ARCHIVE_CDKEY_2')
archive_path_d2char=$(archive_path 'ARCHIVE_D2CHAR')
mkdir --parents "${PLAYIT_WORKDIR}/gamedata"
information_archive_data_extraction "$archive_path"
(
	cd "${PLAYIT_WORKDIR}/gamedata"
	smpq --extract "$archive_path"

	# Update the .mpq file
	cp --dereference "$archive_path_d2char" ./Common/
	(
		cd 'PC-100x'
		smpq --append --overwrite ../Common/d2char.mpq ./*
	)
	(
		cd 'Files'
		mkdir --parents \
			'data/global/sfx/cursor' \
			'data/global/chars/am/cof'
		cp --dereference \
			"$archive_path_cdkey_1" \
			'data/global/sfx/cursor'
		cp --dereference \
			"$archive_path_cdkey_2" \
			'data/global/chars/am/cof'
		smpq --append --overwrite ../Common/d2char.mpq \
			data/global/sfx/cursor/curindx.wav \
			data/global/chars/am/cof/amblxbow.cof
		rm --force --recursive 'data/global/sfx' 'data/global/chars/am'
	)
)

# Include game data

content_inclusion_icons 'PKG_DATA_SHARED' 'APP_MAIN'
content_inclusion_default

# Build packages

packages_generation
print_instructions

# Clean up

working_directory_cleanup

exit 0
